AgentBuilderPreview = {
    
    /**
     * Prepare the preview page elements and events
     * 
     * @param {type} jsonDef
     * @param {type} options
     * @param {type} inputsPropertiesOptions
     * @returns {undefined}
     */
    init : function(jsonDef, options, inputsPropertiesOptions) {
        AgentBuilderPreview.blocks = [];
        AgentBuilderPreview.connections = [];
        AgentBuilderPreview.activeNodeId = null;
        AgentBuilderPreview.initTabButtonsHandler();
        AgentBuilderPreview.initGuider();
        AgentBuilderPreview.renderInputs(jsonDef, options, inputsPropertiesOptions);
        $("#runAgent").off('click')
                .on("click", function(){
                    $(this).addClass("loading");
                    $(this).prepend('<i class="las la-spinner la-spin" style="padding:0 3px;"></i>');
                    $(this).prop("disabled", true);
                    $(this).css("opacity", "0.8");
                    $(".agent-result").hide();
                    var editor = $($('.agent-inputs-container')).data("editor");
                    editor.save();
                });
    },

    /**
     * Render the inputs section with properties editor
     *
     * @param {type} jsonDef
     * @param {type} options
     * @param {type} inputsPropertiesOptions
     * @returns {undefined}
     */
    renderInputs : function(jsonDef, options, inputsPropertiesOptions) {
        // render properties
        var poptions = {
            appPath: options.appPath, // use directly from AgentBuilderPreview.init
            contextPath: options.contextPath,
            propertiesDefinition : inputsPropertiesOptions,
            changeCheckIgnoreUndefined: true,
            saveCallback: function(container, properties) {
                var inputJson = JSON.encode(properties);

                //run the agent if the save is from button clicked
                if ($("#runAgent").hasClass("loading")) {
                    const runId = crypto.randomUUID();
                    console.log("runid: ", runId)
                    options.runId = runId;
                    AgentBuilderPreview.clearResponse();
                    AgentBuilderPreview.listenAgentTraceEvent(options)
                        .finally(() => {
                            AgentBuilderPreview.runAgent(jsonDef, options, inputJson);
                        });
                }
            },
            validationFailedCallback: function() {
                $("#runAgent").removeClass("loading");
                $("#runAgent").find("i").remove();
                $("#runAgent").prop("disabled", false);
                $("#runAgent").css("opacity", "1");
            }
        };
        PropertyEditor.SimpleMode.render($('.agent-inputs-container'), poptions);
        console.log("appPath used:", poptions.appPath);

        // Get value rows from jsonDef (not from poptions)
        let inputRows = [];
        try {
            const defObj = typeof jsonDef === 'string' ? JSON.parse(jsonDef) : jsonDef;
            inputRows = defObj.properties?.inputs || [];
        } catch (e) {
            console.error("Invalid jsonDef format", e);
        }

        // Fix file input dialog height issue
        $('.agent-inputs-container [name]').each(function() {
            const inputName = $(this).attr('name');
            const inputWrapper = $(this).closest('.property-editor-property');

            for (const row of inputRows) {
                if (row.name === inputName && row.type === 'file') {
                    const fileInputId = inputName + '_file';
                }
            }
        });
    },

    /**
     * Make an AJAX call to retrive agent response
     *
     * @param {type} jsonDef
     * @param {type} options
     * @param {type} inputs
     * @returns {undefined}
     */
    runAgent : function(jsonDef, options, inputs) {
        var jsonFile = new Blob([jsonDef], {type : 'text/plain'});
        var params = new FormData();
        params.append("jsonFile", jsonFile);
        params.append("inputs", inputs);

        $.ajax({
            type: "POST",
            url: options.url + "&runId=" + options.runId,
            data: params,
            cache: false,
            processData: false,
            contentType: false,
            dataType: "json",
            timeout: options.timeout,
            beforeSend: function (request) {
               request.setRequestHeader(ConnectionManager.tokenName, ConnectionManager.tokenValue);
            },
            success:function(data) {
                //rendering the response
                AgentBuilderPreview.populate("agent-result-content", data?.content || "", "text");
                AgentBuilderPreview.populate("agent-result-request-payload", data?.requestPayload || "{}");
                AgentBuilderPreview.populate("agent-result-full-result", data?.fullResponse || "{}");
            },
            error:function(data) {
                AgentBuilderPreview.populate("agent-result-content", options?.err, "text");
                AgentBuilderPreview.populate("agent-result-request-payload", "");
                AgentBuilderPreview.populate("agent-result-full-result", "");
            },
            complete: function() {
                //show the result section and enable back the button
                $(".agent-result").show();
                $("#runAgent").removeClass("loading");
                $("#runAgent").find("i").remove();
                $("#runAgent").prop("disabled", false);
                $("#runAgent").css("opacity", 1);
            }
        });
    },

    /**
     * Populate the result to preview page using a code editor
     *
     * @param {type} id
     * @param {type} result
     * @param {type} mode
     * @returns {undefined}
     */
    populate: function(id, result, mode) {
        $("."+id).find(".code-editor").remove();
        $("."+id).append('<div id="code_'+id+'" class="code-editor"></div>');

        // Defensive defaulting
        if (result == undefined || result == null) result = "";

        if (mode === undefined) {
            mode = "application/json";
        }

        // Prepare prettified JSON for later toggling
        var isPureJson = false;
        var prettyJsonResult = null;
        try {
            var parsed = JSON.parse(result);
            // Classify as pure JSON if result is object or array
            if (typeof parsed === "object" && parsed !== null) {
                prettyJsonResult = JSON.stringify(parsed, null, 4);
                isPureJson = true;
            }
        } catch (e) {
            // Do nothing if not pure JSON
        }

        var codeMirror = CodeMirror(document.getElementById("code_"+id), {
            lineNumbers: true,
            mode: mode,
            matchBrackets: true,
            theme: "default",
            autoRefresh:true,
            gutters: ["CodeMirror-lint-markers", "CodeMirror-linenumbers", "CodeMirror-foldgutter"],
            lint: true,
            historyEventDelay: 100,
            autoCloseTags: true,
            autoCloseBrackets: true,
            foldGutter: true,
            lineWrapping: true,
            readOnly: true,
            highlightSelectionMatches: {annotateScrollbar: true, minChars: 1}
        });

        codeMirror.setValue(result);

        // Append checkbox in "Content" section (if result is pure JSON)
        if (id === "agent-result-content" && isPureJson) {
            $('.prettyJsonToggle input[type="checkbox"]').on("change", function() {
                codeMirror.setValue(this.checked ? prettyJsonResult : result);
            });
        } else {
            $("."+id).find(".prettyJsonToggle").remove();
        }

    },

    connectBlocks: function (fromDiv, fromPort, toDiv, toPort, svg, fromId, toId) {
        const fromConnector = fromDiv.querySelector(`.connector.${fromPort}`);
        const toConnector = toDiv.querySelector(`.connector.${toPort}`);
        if (fromConnector && toConnector) {
            const fromPos = this.getConnectorPosition(fromConnector, svg);
            const toPos = this.getConnectorPosition(toConnector, svg);
            const line = this.createLine(fromPos.x, fromPos.y, toPos.x, toPos.y, svg);
            AgentBuilderPreview.connections.push({
                from: { blockId: fromId, port: fromPort },
                to: { blockId: toId, port: toPort },
                line: line
            });
        }
    },

    connectTasks: function (svg) {
        const taskBlocks = AgentBuilderPreview.blocks.filter(b => b.element.classList.contains("task-block"));
        for (let i = 0; i < taskBlocks.length - 1; i++) {
            const fromBlock = taskBlocks[i];
            const toBlock = taskBlocks[i + 1];
            this.connectBlocks(fromBlock.element, "right", toBlock.element, "left", svg, fromBlock.id, toBlock.id);
        }
    },

    extractLabelFromClassName: function (fullClassName) {
        if (typeof fullClassName !== "string") return "";
        return fullClassName.split(".").pop().replace(/([a-z])([A-Z])/g, '$1 $2')
            .replace(/([A-Z]+)([A-Z][a-z])/g, '$1 $2')
            .trim();
    },

    // Update all connections for a block by its ID
    updateNodeConnections: function (blockId) {
        const svg = document.getElementById("connection-layer");
        AgentBuilderPreview.connections.forEach(conn => {
            if (conn.from.blockId === blockId || conn.to.blockId === blockId) {
                const fromBlock = AgentBuilderPreview.blocks.find(b => b.id === conn.from.blockId);
                const toBlock = AgentBuilderPreview.blocks.find(b => b.id === conn.to.blockId);
                if (!fromBlock || !toBlock) return;

                const fromConnector = fromBlock.element.querySelector(`.connector.${conn.from.port}`);
                const toConnector = toBlock.element.querySelector(`.connector.${conn.to.port}`);
                if (!fromConnector || !toConnector) return;

                const fromPos = this.getConnectorPosition(fromConnector, svg);
                const toPos = this.getConnectorPosition(toConnector, svg);

                if (conn.line) {
                    const curvature = 0.5;
                    const dx = toPos.x - fromPos.x;
                    const dy = toPos.y - fromPos.y;
                    let c1x, c1y, c2x, c2y;

                    if (Math.abs(dx) > Math.abs(dy)) {
                        c1x = fromPos.x + dx * curvature;
                        c1y = fromPos.y;
                        c2x = toPos.x - dx * curvature;
                        c2y = toPos.y;
                    } else {
                        c1x = fromPos.x;
                        c1y = fromPos.y + dy * curvature;
                        c2x = toPos.x;
                        c2y = toPos.y - dy * curvature;
                    }

                    conn.line.setAttribute(
                        "d",
                        `M${fromPos.x},${fromPos.y} C${c1x},${c1y} ${c2x},${c2y} ${toPos.x},${toPos.y}`
                    );
                }
            }
        });
    },

    // Setup connector drag handlers
    setupConnectorDrag: function (svg, canvas) {
        const self = this;

        $(document).on("mousedown", ".connector.out", function (e) {
            e.stopPropagation();
            self.isDraggingConnector = true;
            self.fromConnector = this;
            const pos = self.getConnectorPosition(self.fromConnector, svg);
            self.tempLine = self.createLine(pos.x, pos.y, pos.x, pos.y, svg);
        });

        $(document).on("mousemove", function (e) {
            if (self.tempLine && self.isDraggingConnector) {
                const svgRect = svg.getBoundingClientRect();
                const x = e.clientX - svgRect.left;
                const y = e.clientY - svgRect.top;
                self.tempLine.setAttribute("x2", x);
                self.tempLine.setAttribute("y2", y);
            }
        });

        $(document).on("mouseup", function (e) {
            if (self.tempLine && self.isDraggingConnector) {
                const $target = $(e.target);
                if ($target.hasClass("connector") && $target.hasClass("in")) {
                    const toConnector = e.target;
                    const fromBlockId = $(self.fromConnector).closest(".block").data("id");
                    const toBlockId = $(toConnector).closest(".block").data("id");

                    if (fromBlockId === toBlockId) {
                        $(self.tempLine).remove();
                        self.tempLine = null;
                        self.fromConnector = null;
                        self.isDraggingConnector = false;
                        return;
                    }

                    const fromPos = self.getConnectorPosition(self.fromConnector, svg);
                    const toPos = self.getConnectorPosition(toConnector, svg);

                    self.tempLine.setAttribute("x1", fromPos.x);
                    self.tempLine.setAttribute("y1", fromPos.y);
                    self.tempLine.setAttribute("x2", toPos.x);
                    self.tempLine.setAttribute("y2", toPos.y);

                    self.connections.push({
                        from: { blockId: fromBlockId, port: "out" },
                        to: { blockId: toBlockId, port: "in" },
                        line: self.tempLine
                    });
                } else {
                    $(self.tempLine).remove();
                }
                self.tempLine = null;
                self.fromConnector = null;
                self.isDraggingConnector = false;
            }
        });
    },

    // Helper to get connector position relative to SVG
    getConnectorPosition: function (connector, svg) {
        const rect = connector.getBoundingClientRect();
        const svgRect = svg.getBoundingClientRect();
        return {
            x: rect.left + rect.width / 2 - svgRect.left,
            y: rect.top + rect.height / 2 - svgRect.top
        };
    },

    // Create SVG line and append
    createLine: function (x1, y1, x2, y2, svg) {
        const curvature = 0.5;
        const dx = x2 - x1;
        const dy = y2 - y1;
        let c1x, c1y, c2x, c2y;

        if (Math.abs(dx) > Math.abs(dy)) {
            // Horizontal layout: control points offset horizontally
            c1x = x1 + dx * curvature;
            c1y = y1;
            c2x = x2 - dx * curvature;
            c2y = y2;
        } else {
            // Vertical layout: control points offset vertically
            c1x = x1;
            c1y = y1 + dy * curvature;
            c2x = x2;
            c2y = y2 - dy * curvature;
        }

        const path = document.createElementNS("http://www.w3.org/2000/svg", "path");
        path.setAttribute(
            "d",
            `M${x1},${y1} C${c1x},${c1y} ${c2x},${c2y} ${x2},${y2}`
        );
        path.setAttribute("stroke", "#00c6ff");
        path.setAttribute("stroke-width", "2");
        path.setAttribute("fill", "none");
        path.setAttribute("z-index", "1");
        svg.appendChild(path);
        return path;
    },


    // Make block draggable and update connections on move
    makeDraggable: function (block) {
        let isDragging = false;
        let shiftX, shiftY;

        block.addEventListener("mousedown", (e) => {
            if (e.target.classList.contains("connector") || this.isDraggingConnector) return;

            isDragging = true;
            const rect = block.getBoundingClientRect();
            const containerRect = document.getElementById("canvas").getBoundingClientRect();
            shiftX = e.clientX - rect.left;
            shiftY = e.clientY - rect.top;

            e.preventDefault();

            const onMouseMove = (e) => {
                if (!isDragging) return;

                let newLeft = e.clientX - containerRect.left - shiftX;
                let newTop = e.clientY - containerRect.top - shiftY;

                newLeft = Math.max(0, Math.min(newLeft, containerRect.width - rect.width));
                newTop = Math.max(0, Math.min(newTop, containerRect.height - rect.height));

                block.style.left = `${newLeft}px`;
                block.style.top = `${newTop}px`;

                const blockObj = AgentBuilderPreview.blocks.find(b => b.element === block);
                if (blockObj) {
                    blockObj.position.x = newLeft;
                    blockObj.position.y = newTop;
                }

                this.updateNodeConnections(block.dataset.id);
            };

            const onMouseUp = () => {
                isDragging = false;
                document.removeEventListener("mousemove", onMouseMove);
                document.removeEventListener("mouseup", onMouseUp);
            };

            document.addEventListener("mousemove", onMouseMove);
            document.addEventListener("mouseup", onMouseUp);
        });

        block.ondragstart = () => false;
    },

    updateNodeResponse: function (id) {
        if (AgentBuilderPreview.activeNodeId) {
            const prevBlock = AgentBuilderPreview.blocks.find(b => b.id === AgentBuilderPreview.activeNodeId);
            if (prevBlock && prevBlock.element) {
                prevBlock.element.classList.remove("block-active");
            }
        }

        AgentBuilderPreview.activeNodeId = id;
        const block = AgentBuilderPreview.blocks.find(b => b.id === id);
        if (block && block.element) {
            block.element.classList.add("block-active");
        }
        const req = block.request ? JSON.stringify(block.request, null, 4) : "";
        const res = block.response ? JSON.stringify(block.response, null, 4) : "";

        if (block) {
            AgentBuilderPreview.populate(
                "node-result-request-payload",
                req,
                AgentBuilderPreview.isJsonString(req) ? "application/json" : "text"
            );
            AgentBuilderPreview.populate(
                "node-result-full-result",
                res,
                AgentBuilderPreview.isJsonString(res) ? "application/json" : "text"
            );
        }
    },

    clearResponse: function () {
        AgentBuilderPreview.populate(
            "node-result-request-payload",
            "",
            "text"
        );
        AgentBuilderPreview.populate(
            "node-result-full-result",
            "",
            "text"
        );
    },

    getNodeIcon: function (type) {
        switch (type) {
            case "llm": return "fas fa-robot";
            case "prompt": return "fas fa-comment-dots";
            case "tool": return "fas fa-wrench";
            case "task": return "fas fa-tasks";
            case "end": return "fas fa-stop";
            case "enhancer": return "fas fa-magic";
            default: return "fas fa-cube";
        }
    },

    createNode: function ({ id, type, left, top, label, request, response }) {
        const div = document.createElement("div");
        div.classList.add("block", `${type.toLowerCase()}-block-wrapper`);
        div.dataset.id = id;
        div.style.left = left + "px";
        div.style.top = top + "px";

        div.innerHTML = `
            <div class="block-content ${type.toLowerCase()}-block"  id="block-${id}">
                <i class="${AgentBuilderPreview.getNodeIcon(type.toLowerCase())} block-icon"></i>
                <div class="block-type">${type}</div>
                <div class="block-label">${label}</div>
            </div>
        `;

        // Add connectors
        if (type === "Task") {
            ["right"].forEach(name => {
                const conn = document.createElement("div");
                conn.classList.add("connector", name);
                div.appendChild(conn);
            });
        } else if (type === "End") {
            ["left"].forEach(name => {
                const conn = document.createElement("div");
                conn.classList.add("connector", name);
                div.appendChild(conn);
            });
        } else {
            ["left", "right"].forEach(name => {
                const conn = document.createElement("div");
                conn.classList.add("connector", name);
                div.appendChild(conn);
            });
        }
        div.onclick = function () {
            AgentBuilderPreview.updateNodeResponse(id);
        };
        div.classList.add("block-appear");
        div.addEventListener('animationend', () => {
            div.classList.remove("block-appear");
            AgentBuilderPreview.updateNodeConnections(id);
        });
        return div;
    },

    updateNodeState: function (nodeId, state) {
        const node = document.querySelector(`.block[data-id="${nodeId}"]`);
        if (!node) return;

        function getStateClass(base) {
            if (state === "success") return base + "-success";
            if (state === "failed") return base + "-failed";
            if (state === "running") return base + "-running";
            return "";
        }

        // Update node block class
        node.classList.remove("block-success", "block-failed", "block-running", "loading");
        if (state === "success") node.classList.add("block-success");
        else if (state === "failed") node.classList.add("block-failed");
        else if (state === "running") node.classList.add("block-running", "loading");

        // Update connectors and lines for all connections
        AgentBuilderPreview.connections.forEach(conn => {
            const fromBlock = document.querySelector(`.block[data-id="${conn.from.blockId}"]`);
            const toBlock = document.querySelector(`.block[data-id="${conn.to.blockId}"]`);

            function getNodeState(block) {
                if (!block) return "";
                if (block.classList.contains("block-success")) return "success";
                if (block.classList.contains("block-failed")) return "failed";
                if (block.classList.contains("block-running")) return "running";
                return "";
            }

            // Update connectors based on their node's state
            if (fromBlock) {
                const fromConnector = fromBlock.querySelector(`.connector.${conn.from.port}`);
                if (fromConnector) {
                    fromConnector.classList.remove("connector-success", "connector-failed", "connector-running");
                    const fromState = getNodeState(fromBlock);
                    if (fromState) fromConnector.classList.add(getStateClass("connector").replace(state, fromState));
                }
            }
            if (toBlock) {
                const toConnector = toBlock.querySelector(`.connector.${conn.to.port}`);
                if (toConnector) {
                    toConnector.classList.remove("connector-success", "connector-failed", "connector-running");
                    const toState = getNodeState(toBlock);
                    if (toState) toConnector.classList.add(getStateClass("connector").replace(state, toState));
                }
            }

            // Update line
            if (conn.line) {
                conn.line.classList.remove("line-success", "line-failed", "line-running");
                const fromState = getNodeState(fromBlock);
                const toState = getNodeState(toBlock);
                if (fromState === "failed" || toState === "failed") {
                    conn.line.classList.add("line-failed");
                } else if (fromState === "running" || toState === "running") {
                    conn.line.classList.add("line-running");
                } else if (fromState === "success" && toState === "success") {
                    conn.line.classList.add("line-success");
                }
            }
        });
    },

    resetCanvas: function () {
        const canvas = document.getElementById("canvas");
        const svg = document.getElementById("connection-layer");
        canvas.innerHTML = "";
        svg.innerHTML = "";
        this.blocks = [];
        this.canvas = canvas;
        this.svg = svg;
    },

    listenAgentTraceEvent: function (options) {
        if (window.agentSSE) {
            try { window.agentSSE.close(); } catch (e) { }
        }

        this.resetCanvas();

        return new Promise((resolve, reject) => {
            const url = `${options.urlTrace}?runId=${options.runId}`;
            const evtSource = new EventSource(url, { withCredentials: true });
            window.agentSSE = evtSource;

            const context = { currentTask: null, horizontalIndex: 0, lastNode: null, parentMap: {} };

            evtSource.addEventListener("agent-trace-event", (event) => {
                try {
                    const payload = JSON.parse(event.data);
                    AgentBuilderPreview.onTracePayload(payload, context);
                } catch (e) {
                    console.warn("Could not parse SSE payload:", e, event && event.data);
                }
            });

            evtSource.onopen = () => {
                console.log("SSE connected");
                resolve(true);
            };

            evtSource.onerror = (ev) => {
                console.error("SSE failed: ", ev);
                try { evtSource.close(); } catch (e) { }
                resolve(false);
            };

            evtSource.addEventListener("end", () => AgentBuilderPreview.closeSSE(evtSource));

        });
    },

    onTracePayload: function (payload, context) {
        const nodeId = payload.nodeId;
        const parentId = payload.parentId;
        const type = payload.type;
        let status = payload.status;
        const request = payload.request;
        const response = payload.response;

        if (status === "RUNNING") {
            this.handleRunningEvent(payload, { nodeId, type, status, request, response, parentId }, context);
        } else if (status === "END") {
            this.handleRunningEvent(payload, { nodeId, type, status, request, response, parentId }, context);
            this.updateNodeState(nodeId, "SUCCESS".toLowerCase());
            AgentBuilderPreview.removeUnusedRightConnectors();
            return;
        } else {
            this.updateNodeData(nodeId, request, response);
        }

        this.updateNodeState(nodeId, status?.toLowerCase());
        if (status === "FAILED" && context.currentTask) {
            this.updateNodeState(context.currentTask.id, "failed");
        }
    },

    handleRunningEvent: function (payload, normalized, context) {
        const { nodeId, type, request, response, parentId, status } = normalized;
        let left, top;


        if (type === "Task") {
            const pos = this.calculateTaskPosition();
            left = pos.left; top = pos.top;
            context.parentMap[nodeId] = {
                lastNode: null,
                lastPosition: { x: left, y: top }
            };
            context.currentTask = { id: nodeId, left, top };
        } else {
            const pos = this.calculateNodePosition(context, parentId);
            left = pos.left;
            top = pos.top;
        }

        const div = this.createNode({
            id: nodeId,
            type,
            left,
            top,
            label: payload.label || payload.type,
            request,
            response
        });
        this.adjustPreviewPanel(left + div.offsetWidth, top + div.offsetHeight);
        this.canvas.appendChild(div);
        this.blocks.push({ id: nodeId, element: div, position: { x: left, y: top }, type, request, response });
        this.makeDraggable(div);

        // Connect to parent node
        if (type !== "Task") {
            const parentData = context.parentMap[parentId];
            if (parentData && parentData.lastNode) {
                this.connectBlocks(parentData.lastNode.element, "right", div, "left", this.svg, parentData.lastNode.id, nodeId);
            } else if (context.currentTask) {
                const parentEl = this.canvas.querySelector(`.block[data-id="${context.currentTask.id}"]`);
                if (parentEl) {
                    this.connectBlocks(parentEl, "right", div, "left", this.svg, context.currentTask.id, nodeId);
                }
            }
        }

        // Update last node and position for this parentId
        if (type === "Task") {
            context.parentMap[nodeId].lastNode = this.blocks.at(-1);
            context.parentMap[nodeId].lastPosition = { x: left, y: top };
        } else if (parentId) {
            if (!context.parentMap[parentId]) context.parentMap[parentId] = {};
            context.parentMap[parentId].lastNode = this.blocks.at(-1);
            context.parentMap[parentId].lastPosition = { x: left, y: top };
        }

    },

    calculateTaskPosition: function () {
        const taskCount = this.blocks.filter(b => b.type === "Task").length;
        return { left: 50, top: 50 + taskCount * 125 };
    },

    calculateNodePosition: function (context, parentId) {
        const parentData = context.parentMap[parentId];

        if (parentData && parentData.lastNode) {
            return { left: parentData.lastPosition.x + 225, top: parentData.lastPosition.y };
        } else if (parentData) {
            return { left: parentData.lastPosition.x + 225, top: parentData.lastPosition.y };
        } else if (context.currentTask) {
            return { left: context.currentTask.left + 225, top: context.currentTask.top };
        } else {
            return { left: 450, top: 100 };
        }
    },

    updateNodeData: function (nodeId, request, response) {
        const block = this.blocks.find(b => b.id === nodeId);
        if (block) {
            block.request = request;
            block.response = response;
        }
    },

    closeSSE: function (evtSource) {
        try { evtSource.close(); } catch (e) { }
        if (window.agentSSE === evtSource) window.agentSSE = null;
        console.log("closed sse connection");
    },

    isJsonString: function (str) {
        if (!str || typeof str != "string") return false;
        str = str.trim();
        return (str.startsWith("{") && str.endsWith("}")) ||
            (str.startsWith("[") && str.endsWith("]")) ||
            (str.startsWith('"') && str.endsWith('"'));
    },

    adjustPreviewPanel: function (nodeRightEdge, nodeBottomEdge) {
        const canvas = document.getElementById("canvas");
        const svg = document.getElementById("connection-layer");

        const currentWidth = canvas.offsetWidth;
        const currentHeight = canvas.offsetHeight;

        const newWidth = Math.max(currentWidth, nodeRightEdge + 50);
        const newHeight = Math.max(currentHeight, nodeBottomEdge + 50);

        canvas.style.width = newWidth + "px";
        svg.style.width = newWidth + "px";

        canvas.style.height = newHeight + "px";
        svg.style.height = newHeight + "px";
    },

    initTabButtonsHandler: function () {
        document.querySelectorAll('.agent-tab-btn').forEach(btn => {
            btn.onclick = () => {
                document.querySelectorAll('.agent-tab-btn').forEach(b => b.classList.remove('active'));
                btn.classList.add('active');
                document.querySelectorAll('.agent-tab-content').forEach(tab => tab.style.display = 'none');
                document.getElementById('agent-tab-' + btn.dataset.tab).style.display = 'block';
            };
        });
    },

    removeUnusedRightConnectors: function () {
        AgentBuilderPreview.blocks.forEach(block => {
            const rightConnector = block.element.querySelector('.connector.right');
            if (rightConnector) {
                const hasOutgoing = AgentBuilderPreview.connections.some(
                    conn => conn.from.blockId === block.id
                );
                if (!hasOutgoing) {
                    rightConnector.remove();
                }
            }
        });
    },

    startGuider: function () {
        if (typeof window.driver === "undefined" || typeof window.driver.js === "undefined") return;
        const driver = window.driver.js.driver;
        const driverObj = driver({
            showProgress: true,
            nextBtnText: 'Next',
            prevBtnText: 'Previous',
            doneBtnText: 'Done',
            showButtons: [
                'next',
                'previous'
            ],
            steps: [
                {
                    element: ".agent-tab-btn[data-tab='execution']",
                    popover: {
                        title: "Execution Tab",
                        description: "Start here to configure your inputs and run the agents.",
                        side: "bottom"
                    }
                },
                {
                    element: ".agent-tab-btn[data-tab='response']",
                    popover: {
                        title: "Result Tab",
                        description: "Switch here to see the final output and responses.",
                        side: "bottom"
                    }
                },
                {
                    element: ".agent-inputs",
                    popover: {
                        title: "Configure Inputs",
                        description: "Define the inputs your agent will use here.",
                        side: "right"
                    }
                },
                {
                    element: ".agent-node-viewer",
                    popover: {
                        title: "Agent node viewer panel",
                        description: "This is the node canvas where your agent workflow is visualized. You can click a node to inspect its request and response details.",
                        side: "left"
                    }
                },
                {
                    element: ".node-result-request-payload",
                    popover: {
                        title: "Node Request",
                        description: "Inspect the request payload of the node here.",
                        side: "left"
                    }
                },
                {
                    element: ".node-result-full-result",
                    popover: {
                        title: "Node Response",
                        description: "Review the response of the node here.",
                        side: "left"
                    }
                }
            ],

            onPopoverRender: (popover, { config, state }) => {
                const disableBtn = document.createElement("button");
                disableBtn.style.display = "block";
                disableBtn.innerText = "Disable Hints";

                popover.footerButtons.appendChild(disableBtn);

                disableBtn.addEventListener("click", () => {
                    localStorage.setItem("agentTourDisabled", "true");
                    driverObj.destroy();
                });
            }

        });

        driverObj.drive();
    },

    initGuider: function () {
        document.addEventListener("DOMContentLoaded", function () {
            const isTourDisabled = localStorage.getItem("agentTourDisabled");
            if (!isTourDisabled) {
                AgentBuilderPreview.startGuider();
            }
        });
    },
};
