/*!
 * AgentInjectorPopUp jQuery Plugin (OOP, Sectioned)
 * -------------------------------------------
 *
 * TABLE OF CONTENTS
 * [1] CLASS DEFINITION: AgentInjectorPopUp
 *   [1.1] Constructor & Initialization
 *   [1.2] DOM Caching
 *   [1.3] Event Binding
 * [2] WEBSOCKET HANDLING
 *   [2.1] Init/Connect WebSocket
 *   [2.2] Send Message on WebSocket
 *   [2.3] Handle Incoming WebSocket Messages
 * [3] UI INTERACTION & UPDATE
 *   [3.1] Update Connection Status
 *   [3.2] Message List: Add/Scroll/Step Actions
 *   [3.3] Send/Receive/Typing Controls
 * [4] CONTENT FORMATTING AND UTILS
 *   [4.1] Step Content Formatting
 *   [4.2] Text Escape
 *   [4.3] Typing Dots Animation
 * [5] JQUERY PLUGIN WRAPPER
 */

(function ($) {

    // [1] CLASS DEFINITION: AgentInjectorPopUp
    class AgentInjectorPopUp {
        // [1.1] Constructor & Initialization
        constructor(el, options) {
            this.options = options || {};
            this.chatEl = $(el);
            this.sessionId = 'session_' + Math.random().toString(36).substr(2, 9);
            this.ws = null;
            this.isConnected = false;
            this.isInitialized = false;
            this.lock = false;

            // [1.2] DOM Caching
            this.cacheDom();

            // [1.3] Event Binding
            this.bindEvents();

            // [2.1] Init/Connect WebSocket on load
            this.initWebSocket();

            // Cleanup on window unload
            $(window).on('beforeunload', () => {
                if (this.ws) this.ws.close();
            });
        }

        // [1.2] Cache all DOM nodes for easy/fast access
        cacheDom() {
            this.dom = {
                chatIcon:    this.chatEl.find('#chatIcon'),
                chatPopup:   this.chatEl.find('#chatPopup'),
                sendButton:  this.chatEl.find('#chat-send-btn'),
                chatForm:    this.chatEl.find('#chat-form'),
                userInput:   this.chatEl.find('#user-input'),
                messages:    this.chatEl.find('#messages'),
                messagesCon: this.chatEl.find('.msger-messages'),
                headerTitle: this.chatEl.find('#msger-header-title')
            };
        }

        // [1.3] Wire up all UI events and handlers
        bindEvents() {
            const d = this.dom;

            d.chatIcon.off('click.agentInjectorPopUp').on('click.agentInjectorPopUp', e => {
                d.chatPopup.toggleClass('show');
                if (d.chatPopup.hasClass('show')) {
                    if (!this.ws || this.ws.readyState === WebSocket.CLOSED) {
                        this.initWebSocket();
                    } else if (this.ws.readyState === WebSocket.OPEN && !this.isInitialized) {
                        this.sendWebSocketMessage({ type: 'init', sessionId: this.sessionId });
                        this.isInitialized = true;
                    }
                    d.userInput.focus();
                }
            });

            d.chatForm.off('submit.agentInjectorPopUp').on('submit.agentInjectorPopUp', e => {
                e.preventDefault();
                this.sendMessage();
                return false;
            });

            d.sendButton.off('click.agentInjectorPopUp').on('click.agentInjectorPopUp', e => {
                e.preventDefault();
                this.sendMessage();
            });

            d.userInput.off('keypress.agentInjectorPopUp').on('keypress.agentInjectorPopUp', e => {
                if (e.which === 13 && !e.shiftKey) {
                    e.preventDefault();
                    this.sendMessage();
                }
            });
        }

        // [2.1] WebSocket connection lifecycle & reconnect logic
        initWebSocket() {
            try {
                const protocol = window.location.protocol === "https:" ? "wss://" : "ws://"; // ws: for http, wss: for https
                const wsUrl = protocol + window.location.host + this.options.contextPath + "/web/socket/plugin/org.joget.ai.designer.feature.AgentInjectorPopUp";
                this.ws = new WebSocket(wsUrl);

                this.ws.onopen = () => {
                    this.log('WebSocket connected');
                    this.isConnected = true;
                    this.isInitialized = false;
                    this.updateConnectionStatus(true);
                    this.sendWebSocketMessage({ type: 'init', sessionId: this.sessionId });
                    this.isInitialized = true;
                };
                this.ws.onmessage = (event) => {
                    this.log('Received message:', event.data);
                    this.handleWebSocketEventData(event.data);
                };
                this.ws.onclose = () => {
                    this.log('WebSocket disconnected');
                    this.isConnected = false;
                    this.isInitialized = false;
                    this.updateConnectionStatus(false);
                    setTimeout(() => {
                        if (!this.isConnected) {
                            this.log('Attempting to reconnect...');
                            this.initWebSocket();
                        }
                    }, 3000);
                };
                this.ws.onerror = (error) => {
                    this.isConnected = false;
                    this.isInitialized = false;
                    this.updateConnectionStatus(false);
                    console.error('WebSocket error:', error);
                };
            } catch (error) {
                this.appendMessage('Connection failed. Please refresh the page.', 'received error');
            }
        }

        // [2.2] Send data/messages through WebSocket
        sendWebSocketMessage(messageObj) {
            if (this.ws && this.ws.readyState === WebSocket.OPEN) {
                this.log('Sending WebSocket message:', JSON.stringify(messageObj));
                this.ws.send(JSON.stringify(messageObj));
                if (messageObj.type === 'init') this.dom.messages?.empty(); // Clear messages on init
                return true;
            }
            this.appendMessage('Connection lost. Trying to reconnect...', 'received error');
            return false;
        }

        // [2.3] WebSocket: Incoming messages & error handling
        handleWebSocketEventData(data) {
            // Always clear waiting when a new message comes in
            this.lock = false;
            this.dom.userInput.prop('disabled', this.lock);
            this.dom.sendButton.prop('disabled', this.lock).text('Send');

            this.hideTypingIndicator();
            try {
                const message = JSON.parse(data);
                this.handleWebSocketMessage(message);
            } catch (e) {
                this.appendMessage(data, 'received');
            }
        }

        // [2.3] WebSocket: Dispatch specific message types
        handleWebSocketMessage(message) {
            switch (message.type) {
                case 'step_1_proposal':
                case 'step_2_structure':
                case 'step_3_joget_agent':
                    this.appendMessage(this.formatStepContent(message.content, message.step), 'received');
                    this.appendStepActions(message.actions);
                    break;
                case 'chat_response':
                    message.content
                        ? this.appendMessage(message.content, 'received')
                        : this.appendMessage('Error: ' + message.error, 'received error');
                    break;
                case 'error':
                    this.appendMessage('Error: ' + (message.message || 'Unknown error occurred'), 'received error');
                    break;
                case 'status':
                    if (message.message) this.log('Status:', message.message);
                    break;
                default:
                    const content = message.content || message.response || message.answer;
                    this.appendMessage(content ? content : JSON.stringify(message), 'received');
            }
        }

        // [3.1] UI: Indicate connection (header/status)
        updateConnectionStatus(connected) {
            this.dom.headerTitle.html(
                '<i class="fas fa-robot"></i> AI Designer - Agent Injector ' +
                `<span class="status-indicator ${connected ? 'connected' : 'disconnected'}"></span>`
            );
        }

        // [3.2] Message List: Add/Scroll/Step Actions
        appendMessage(message, className) {
            const el = $('<li class="message ' + className + '"></li>');
            if (className.includes('received') && message && message.includes && message.includes('<')) {
                el.html(message);
            } else {
                el.text(message);
            }
            this.dom.messages.append(el);
            this.scrollToBottom();
        }

        appendStepActions(actions) {
            if (!actions || actions.length === 0) return;
            const actionsHtml = $('<div class="step-actions"></div>');
            actions.forEach(a => {
                $('<button class="step-action-btn"></button>')
                    .text(a.charAt(0).toUpperCase() + a.slice(1))
                    .on('click', () => {
                        // Remove ALL action button messages when ANY action is clicked
                        this.dom.messages.find('li.message.action').remove();
                        // Lock UI
                        this.lock = true;
                        this.dom.userInput.prop('disabled', this.lock);
                        this.dom.sendButton.prop('disabled', this.lock);

                        this.showTypingIndicator();
                        this.sendWebSocketMessage({
                            type: a === "continue" ? "continue_step" : "init",
                            sessionId: this.sessionId
                        });
                    })
                    .appendTo(actionsHtml);
            });
            this.dom.messages.append($('<li class="message action"></li>').append(actionsHtml));
            this.scrollToBottom();
        }

        // [3.2] Scroll Chat to Bottom Helper
        scrollToBottom = () => this.dom.messagesCon.scrollTop(this.dom.messagesCon[0].scrollHeight);

        // [3.3] UI: Sending Logic & Typing Indicator
        sendMessage() {
            const d = this.dom;
            const text = d.userInput.val().trim();
            if (!text || !this.isConnected) {
                if (!this.isConnected) this.appendMessage('Not connected. Please wait for connection...', 'received error');
                return;
            }
            
            if (this.lock) return; // Prevent sending while waiting
            this.lock = true;

            d.userInput.val('');
            d.userInput.prop('disabled', this.lock);
            d.sendButton.prop('disabled', this.lock).text('Sending...');
            this.appendMessage(text, 'sent');
            this.showTypingIndicator();

            const messageData = {
                type: 'chat_message',
                message: text,
                sessionId: this.sessionId,
                timestamp: new Date().toISOString()
            };

            if (!this.sendWebSocketMessage(messageData)) {
                this.hideTypingIndicator();
                this.appendMessage('Failed to send message. Please try again.', 'received error');
            }
        }

        // [3.3] Typing Loader
        showTypingIndicator() {
            this.dom.messages.append('<li class="message received typing" id="typing-indicator">AI is typing<span class="dots">...</span></li>');
            this.scrollToBottom();
            this.animateTypingDots();
        }
        hideTypingIndicator = () => this.dom.messages.find('#typing-indicator').remove(); 
        
        // [4.3] Typing Animation Dots (UI flair)
        animateTypingDots() {
            const dots = this.dom.messages.find('#typing-indicator .dots');
            if (dots.length) {
                let dotCount = 0;
                const interval = setInterval(() => {
                    if (!this.dom.messages.find('#typing-indicator').length) return clearInterval(interval);
                    dotCount = (dotCount % 3) + 1;
                    dots.text('.'.repeat(dotCount));
                }, 500);
            }
        }

        // [4.1] Format main content of each step (for appendMessage)
        formatStepContent(content, step) {
            if (step == 1) {
                const prefix = '<b>Agent Workflow Proposal</b><br>The workflow of the agent consists of:';
                if (Array.isArray(content)) {
                    return prefix + '<ol>' + content.map(item => '<li>' + AgentInjectorPopUp.escapeHtml(item) + '</li>').join('') + '</ol>';
                } else if (typeof content === 'object' && content !== null) {
                    return prefix + '<pre>' + AgentInjectorPopUp.escapeHtml(JSON.stringify(content, null, 2)) + '</pre>';
                }
                return AgentInjectorPopUp.escapeHtml(content + "");
            } else if (step == 2) {
                let structure = content;
                if (structure && structure.proposed_workflow_parsed) {
                    structure = structure.proposed_workflow_parsed;
                }
                let html = '';
                if (structure && typeof structure === "object") {
                    if (structure.name) html += '<b>Agent Name:</b> ' + AgentInjectorPopUp.escapeHtml(structure.name) + '<br>';
                    if (structure.description) html += '<b>Description:</b> ' + AgentInjectorPopUp.escapeHtml(structure.description) + '<br>';
                    if (Array.isArray(structure.tasks)) {
                        html += '<b>Task Steps:</b><ol>';
                        structure.tasks.forEach(task => {
                            if (task.label) html += '<li>' + AgentInjectorPopUp.escapeHtml(task.label) + '</li>';
                        });
                        html += '</ol>';
                    }
                    if (Array.isArray(structure.inputs)) {
                        html += '<b>Required Inputs:</b><ul>';
                        structure.inputs.forEach(input => {
                            if (input.label) html += '<li>' + AgentInjectorPopUp.escapeHtml(input.label) + '</li>';
                        });
                        html += '</ul>';
                    }
                    return html;
                }
                return '<pre>' + AgentInjectorPopUp.escapeHtml(JSON.stringify(content, null, 2)) + '</pre>';
            } else if (step == 3) {
                let html = '';
                if (content.successMessage) {
                    html += '<div class="agent-success">' + AgentInjectorPopUp.escapeHtml(content.successMessage) + '</div>';
                }
                if (content.agentJson) {
                    setTimeout(() => {
                        if (typeof CustomBuilder !== "undefined" && typeof CustomBuilder.loadJson === "function") {
                            CustomBuilder.loadJson(content.agentJson.agent_structure, true);
                        }
                    }, 500);
                }
                return html;
            }
        }

        // [4.2] Escape HTML to prevent XSS
        static escapeHtml = (str) => $('<div/>').text(str).html();

        log = (...args) => { if (this.options.debug) console.log(...args); }
        
    }

    // [5] jQuery Plugin Wrapper (singleton for each root)
    $.fn.agentInjectorPopUp = function (options) {
        return this.each(function () {
            if (!$.data(this, 'plugin_agentInjectorPopUp')) {
                $.data(this, 'plugin_agentInjectorPopUp', new AgentInjectorPopUp(this, options));
            }
        });
    };

})(jQuery);