/*!
 * CentralConfig jQuery Plugin (OOP, Multi-Tab)
 * -------------------------------------------
 *
 * TABLE OF CONTENTS
 * [1] CLASS DEFINITION: CentralConfig
 *   [1.1] Constructor & Initialization
 *   [1.2] DOM Caching
 *   [1.3] Event Binding
 * [2] API SERVER CONFIGURATION
 *   [2.1] Load/Save API Server Config
 *   [2.2] URL/Key Validation
 *   [2.3] Connection Testing
 * [3] AI SERVICE MANAGEMENT
 *   [3.1] Service Loading & Creation
 *   [3.2] Dynamic Form Generation
 *   [3.3] Service Editing & Deletion
 * [4] UI INTERACTION & UPDATE
 *   [4.1] Tab Management
 *   [4.2] Modal Management
 *   [4.3] Status & Loading States
 * [5] UTILITY METHODS
 *   [5.1] Data Processing
 *   [5.2] Form Helpers
 *   [5.3] State Management
 * [6] JQUERY PLUGIN WRAPPER
 */

(function ($) {

    // [1] CLASS DEFINITION: CentralConfig
    class CentralConfig {
        // [1.1] Constructor & Initialization
        constructor(el, options) {
            this.options = options || {};
            this.configEl = $(el);
            this.data = {
                apiServerConfig: { apiServerUrl: '', apiServerKey: '' },
                services: [],
                currentService: null,
                availableModels: null,
                newAiServiceFormFields: {},
                availableModelOptions: {},
                originalServicesData: [],
                piiMaskingRules: []
            };

            this.cacheDom();
            this.bindEvents();
            this.initialize();
        }

        // [1.2] DOM Caching
        cacheDom() {
            this.dom = {
                // Tabs
                tabButtons: this.configEl.find('.tab-button'),
                tabContents: this.configEl.find('.tab-content'),
                aiServicesTab: this.configEl.find('[data-tab="ai-services"]'),
                apiServerTab: this.configEl.find('[data-tab="api-server"]'),
                
                // API Server Form
                apiServerUrl: this.configEl.find('#api-server-url'),
                apiServerKey: this.configEl.find('#api-server-key'),
                saveConfigBtn: this.configEl.find('#save-config'),
                testConnectionBtn: this.configEl.find('#test-connection'),
                requestKeyBtn: this.configEl.find('#request-joget-key'),
                
                // Services Management
                servicesList: this.configEl.find('#services-list'),
                emptyState: this.configEl.find('#services-empty-state'),
                addServiceBtn: this.configEl.find('#add-service'),
                currentServiceSelect: this.configEl.find('#current-service'),
                
                // Modals
                addModal: this.configEl.find('#add-service-modal'),
                addModalBody: this.configEl.find('#add-service-modal .modal-body'),
                createServiceBtn: this.configEl.find('#create-service'),
                closeModalBtn: this.configEl.find('#close-modal'),
                cancelServiceBtn: this.configEl.find('#cancel-service'),
                
                editModal: this.configEl.find('#edit-service-modal'),
                editModalBody: this.configEl.find('#edit-service-modal .modal-body'),
                editServiceName: this.configEl.find('#edit-service-name'),
                saveEditBtn: this.configEl.find('#save-edit-service'),
                closeEditBtn: this.configEl.find('#close-edit-modal'),
                cancelEditBtn: this.configEl.find('#cancel-edit-service'),

                // PII Masking Rules - Add these lines
                piiMaskingTab: this.configEl.find('[data-tab="pii-masking"]'),
                piiRuleInput: this.configEl.find('#pii-rule-input'),
                addPiiRuleBtn: this.configEl.find('#add-pii-rule'),
                piiRulesList: this.configEl.find('#pii-rules-list'),
                piiRulesEmptyState: this.configEl.find('#pii-rules-empty-state'),
                
                // Status
                status: this.configEl.find('#config-status')
            };
        }

        // [1.3] Event Binding
        bindEvents() {
            // Tab navigation
            this.dom.tabButtons.on('click', e => this.switchTab(e.currentTarget.dataset.tab));
            
            // API Server
            this.dom.saveConfigBtn.on('click', () => this.saveApiServerConfig());
            this.dom.testConnectionBtn.on('click', () => this.testConnection());
            this.dom.requestKeyBtn.on('click', () => window.open('https://intelligence.int.joget.cloud/jw/', '_blank'));
            
            // Services
            this.dom.addServiceBtn.on('click', () => this.showAddServiceModal());
            this.dom.createServiceBtn.on('click', () => this.createService());
            this.dom.currentServiceSelect.on('change', e => this.handleCurrentServiceChange(e.target.value));
            
            // Add Modal
            this.dom.closeModalBtn.on('click', () => this.hideAddServiceModal());
            this.dom.cancelServiceBtn.on('click', () => this.hideAddServiceModal());
            this.dom.addModal.on('click', e => { if (e.target === e.currentTarget) this.hideAddServiceModal(); });
            
            // Edit Modal
            this.dom.closeEditBtn.on('click', () => this.hideEditServiceModal());
            this.dom.cancelEditBtn.on('click', () => this.hideEditServiceModal());
            this.dom.saveEditBtn.on('click', () => this.saveEditedService());
            this.dom.editModal.on('click', e => { if (e.target === e.currentTarget) this.hideEditServiceModal(); });

            // PII Masking Rules
            this.dom.addPiiRuleBtn.on('click', () => this.addPiiRule());
            this.dom.piiRuleInput.on('keypress', e => { 
                if (e.which === 13) this.addPiiRule(); 
            });
            
            // Password toggles
            this.setupPasswordToggle('toggle-server-key', 'api-server-key');
        }

        initialize() {
            this.disableAiServicesTab();
            this.loadApiServerConfig();
        }

        // [2] API SERVER CONFIGURATION
        // [2.1] Load/Save API Server Config
        async loadApiServerConfig() {
            const response = await this.apiCall('loadApiServerConfig');
            if (response?.success && response.apiServerConfig?.properties) {
                const config = response.apiServerConfig.properties;
                this.dom.apiServerUrl.val(config.apiServerUrl || '@@aiag.defaultApiServerUrl@@');
                
                let apiKey = config.apiServerKey || '';
                if (apiKey.startsWith('%%%%') && apiKey.endsWith('%%%%')) {
                    apiKey = apiKey;
                }
                this.dom.apiServerKey.val(apiKey);
                
                this.data.apiServerConfig.apiServerUrl = this.dom.apiServerUrl.val();
                this.data.apiServerConfig.apiServerKey = apiKey;
                
                if (config.apiServerUrl?.trim() && config.apiServerKey?.trim()) {
                    this.enableAiServicesTab();
                }
            } else {
                this.dom.apiServerUrl.val('@@aiag.defaultApiServerUrl@@');
                this.dom.apiServerKey.val('');
            }
        }

        async saveApiServerConfig() {
            const url = this.dom.apiServerUrl.val().trim();
            const key = this.dom.apiServerKey.val().trim();
            
            if (!url || !key) {
                this.showStatus('Please fill in both API Server URL and API Server Key', 'error');
                return;
            }
            
            this.showConfigLoading();
            
            try {
                const correctedUrl = await this.validateAndCorrectUrl(url);
                if (!correctedUrl) {
                    this.showStatus('Invalid API Server URL. Please check and try again.', 'error');
                    setTimeout(() => this.dom.apiServerUrl.val(this.data.apiServerConfig.apiServerUrl), 1000);
                    return;
                }
                
                if (key !== this.data.apiServerConfig.apiServerKey) {
                    const isValidKey = await this.validateApiServerKey(correctedUrl, key);
                    if (!isValidKey) {
                        this.showStatus('Invalid API Server Key. Please check and try again.', 'error');
                        setTimeout(() => this.dom.apiServerKey.val(this.data.apiServerConfig.apiServerKey), 1000);
                        return;
                    }
                }
                
                this.dom.apiServerUrl.val(correctedUrl);
                const success = await this.saveApiServerConfigToServer(correctedUrl, key);
                
                if (success) {
                    this.data.apiServerConfig.apiServerUrl = correctedUrl;
                    this.data.apiServerConfig.apiServerKey = key;
                    this.showStatus('API Server configuration saved successfully', 'success');
                    await this.enableAiServicesTab();
                    
                    if (this.data.services.length === 0) {
                        setTimeout(() => {
                            this.switchTab('ai-services');
                            this.showStatus('You can now configure AI services in the AI Services tab', 'info');
                        }, 1500);
                    }
                } else {
                    this.showStatus('Failed to save configuration. Please try again.', 'error');
                }
            } catch (error) {
                this.showStatus('An error occurred while saving configuration', 'error');
            }
        }

        // [2.2] URL/Key Validation
        async validateAndCorrectUrl(url) {
            let correctedUrl = url.replace(/\/+$/, '');
            if (!correctedUrl.match(/^https?:\/\//)) {
                correctedUrl = 'http://' + correctedUrl;
            }
            
            const response = await this.apiCall('validateApiServerUrl', { '_apiServerUrl': correctedUrl });
            return response === 'true' ? correctedUrl : null;
        }

        async validateApiServerKey(url, key) {
            const response = await this.apiCall('validateApiServerKey', { 
                '_apiServerUrl': url, 
                '_apiServerKey': key 
            });
            return response === 'true';
        }

        async saveApiServerConfigToServer(url, key) {
            const response = await this.apiCall('saveApiServerConfig', { 
                '_apiServerUrl': url, 
                '_apiServerKey': key 
            });
            return response?.success === true;
        }

        // [2.3] Connection Testing
        async testConnection() {
            const url = this.dom.apiServerUrl.val().trim();
            if (!url) {
                this.showStatus('Please enter API Server URL first', 'error');
                return;
            }
            
            const btn = this.dom.testConnectionBtn;
            btn.prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i> Testing...');
            
            try {
                const correctedUrl = await this.validateAndCorrectUrl(url);
                if (correctedUrl) {
                    this.showStatus('Connection test successful', 'success');
                    this.dom.apiServerUrl.val(correctedUrl);
                    this.data.apiServerConfig.apiServerUrl = correctedUrl;
                } else {
                    this.showStatus('Connection test failed - Unable to reach server', 'error');
                }
            } catch (error) {
                this.showStatus('Connection test failed', 'error');
            } finally {
                btn.prop('disabled', false).html('<i class="fas fa-plug"></i> Test Connection');
            }
        }

        // [3] AI SERVICE MANAGEMENT
        // [3.1] Service Loading & Creation
        async loadAvailableModels() {
            const response = await this.apiCall('retrieveNewAiServiceFormFields');
            if (response?.data) {
                this.data.newAiServiceFormFields = response.data;
                response.data.fields.forEach(field => {
                    if (field.id === "model-name") {
                        this.data.availableModelOptions = field.options;
                    }
                });
                return true;
            }
            return false;
        }

        async loadMyAiServices() {
            const response = await this.apiCall('loadMyAiServices');
            if (response?.success && response.myAiServices?.properties) {
                this.updateServicesFromResponse(response.myAiServices.properties.myAiServices);
            } else {
                this.data.services = [];
                this.renderServices();
                this.updateServiceSelect();
            }
            return true;
        }

        loadCurrentAiService() {
            this.apiCall('loadCurrentAiService').then(response => {
                if (response?.success && response.currentAiService?.properties) {
                    const currentServiceName = response.currentAiService.properties.aiService;
                    if (currentServiceName) {
                        const service = this.data.services.find(s => s.name === currentServiceName);
                        if (service) {
                            this.data.currentService = service.id;
                            this.dom.currentServiceSelect.val(service.id);
                            this.data.services.forEach(s => s.active = false);
                            service.active = true;
                            this.renderServices();
                        }
                    }
                }
            });
        }

        createService() {
            const validationErrors = [];
            const createNewAiServiceData = {};
            
            this.clearModalError();
            
            this.dom.addModalBody.find('input, select').each((i, el) => {
                createNewAiServiceData[el.name] = el.value;
            });
            
            const allFields = this.getAllFormFields();
            Object.keys(createNewAiServiceData).forEach(key => {
                const value = createNewAiServiceData[key];
                allFields.forEach(field => {
                    if (key === field.name && field.validation) {
                        const regex = new RegExp(field.validation);
                        if (regex.test(value) && this.shouldActivateValidation(field)) {
                            validationErrors.push(field.validationMessage);
                        }
                    }
                });
            });
            
            if (validationErrors.length > 0) {
                this.showModalError(validationErrors[0]);
                return;
            }
            
            this.showModalLoading('Creating AI service...');
            
            this.apiCall('createNewAiService', { 
                '_createNewAiServiceData': JSON.stringify(createNewAiServiceData) 
            }).then(response => {
                if (response?.success) {
                    this.showStatus('AI Service created successfully!', 'success');
                    this.hideAddServiceModal();
                    if (response.myAiServices?.properties) {
                        this.updateServicesFromResponse(response.myAiServices.properties.myAiServices);
                    }
                } else {
                    this.showModalError('AI Service Name already exists');
                }
            }).catch(() => {
                this.showModalError('Failed to create AI service');
            });
        }

        // [3.2] Dynamic Form Generation
        generateDynamicServiceForm() {
            if (!this.data.newAiServiceFormFields.fields) {
                this.loadAvailableModels().then(() => this.generateDynamicServiceForm());
                return;
            }
            
            this.hideModalLoading();
            this.clearModalLoading();
            this.dom.addModalBody.empty();
            
            this.data.newAiServiceFormFields.fields.forEach(field => {
                if (field.type === "text" || field.type === "password") {
                    this.dom.addModalBody.append(this.createFormGroup(field));
                } else if (field.type === "select") {
                    this.dom.addModalBody.append(this.createSelectGroup(field));
                } else if (field.type === "container") {
                    field.children.forEach(child => {
                        const childFormGroup = this.createFormGroup(child);
                        childFormGroup.attr('data-container', field.id);
                        if (field.hidden) childFormGroup.addClass('hide');
                        this.dom.addModalBody.append(childFormGroup);
                    });
                }
            });
            
            setTimeout(() => {
                const modelClassSelect = this.dom.addModalBody.find('#model-class');
                if (modelClassSelect.length) modelClassSelect.trigger('change');
            }, 100);
        }

        createFormGroup(field) {
            const formGroup = $('<div class="form-group">');
            const label = $(`<label for="${field.id}">${field.label}</label>`);
            const input = $(`<input type="${field.type}" id="${field.id}" name="${field.name}" 
                            class="form-input" placeholder="${field.placeholder || ''}" 
                            autocomplete="${field.type === 'password' ? 'new-password' : 'new-text'}">`);
            
            formGroup.append(label);
            
            if (field.type === 'password') {
                const passwordField = $('<div class="password-field">');
                const toggleBtn = $('<button type="button" class="password-toggle"><i class="fas fa-eye"></i></button>');
                
                toggleBtn.on('click', function() {
                    const icon = $(this).find('i');
                    if (input.attr('type') === 'password') {
                        input.attr('type', 'text');
                        icon.removeClass('fa-eye').addClass('fa-eye-slash');
                    } else {
                        input.attr('type', 'password');
                        icon.removeClass('fa-eye-slash').addClass('fa-eye');
                    }
                });
                
                passwordField.append(input, toggleBtn);
                formGroup.append(passwordField);
            } else {
                formGroup.append(input);
            }
            
            if (field.hidden) formGroup.addClass('hide');
            return formGroup;
        }

        createSelectGroup(field) {
            const formGroup = $('<div class="form-group">');
            const label = $(`<label for="${field.id}">${field.label}</label>`);
            const select = $(`<select id="${field.id}" name="${field.name}" class="form-select">`);
            
            if (Array.isArray(field.options)) {
                field.options.forEach(option => {
                    select.append(`<option value="${option}">${option}</option>`);
                });
            }
            
            if (field.triggers?.includes('change')) {
                select.on('change', () => this.handleFieldChange(field, select.val()));
            }
            
            formGroup.append(label, select);
            if (field.hidden) formGroup.addClass('hide');
            return formGroup;
        }

        handleFieldChange(field, selectedValue) {
            if (field.cascadesTo) {
                const childSelect = $(`#${field.cascadesTo}`);
                if (childSelect.length && this.data.availableModelOptions[selectedValue]) {
                    childSelect.empty();
                    this.data.availableModelOptions[selectedValue].forEach(option => {
                        childSelect.append(`<option value="${option}">${option}</option>`);
                    });
                }
            }
            
            if (field.visibilityDecisions) {
                const ifValues = [];
                
                if (field.visibilityDecisions.if) {
                    field.visibilityDecisions.if.forEach(decision => {
                        ifValues.push(decision.value);
                        if (decision.value === selectedValue) {
                            decision.hide.forEach(toHide => this.toggleElementVisibility(toHide, true));
                            decision.show.forEach(toShow => this.toggleElementVisibility(toShow, false));
                        }
                    });
                }
                
                if (!ifValues.includes(selectedValue) && field.visibilityDecisions.else) {
                    field.visibilityDecisions.else.hide.forEach(toHide => this.toggleElementVisibility(toHide, true));
                    field.visibilityDecisions.else.show.forEach(toShow => this.toggleElementVisibility(toShow, false));
                }
            }
        }

        // [3.3] Service Editing & Deletion
        editService(id) {
            const service = this.data.services.find(s => s.id == id);
            if (!service) {
                this.showStatus('Service not found', 'error');
                return;
            }
            
            this.dom.editModal.addClass('show');
            this.dom.editServiceName.text(service.name);
            this.dom.editModal.attr('data-editing-service-id', id);
            
            this.showEditModalLoading('Loading service details...');
            this.generateEditServiceForm(service);
        }

        generateEditServiceForm(service) {
            const originalService = this.getOriginalServiceData(service.name);
            if (!originalService) {
                this.showEditModalError('Could not load service details');
                return;
            }
            
            this.hideEditModalLoading();
            this.dom.editModalBody.empty();
            
            const allFields = this.getAllFormFields();
            const fieldsToShow = this.getFieldsForModelClass(originalService.modelClass);
            
            allFields.forEach(field => {
                if (fieldsToShow.includes(field.name)) {
                    this.dom.editModalBody.append(this.createEditFormGroup(field, originalService));
                }
            });
        }

        createEditFormGroup(field, originalService) {
            const formGroup = $('<div class="form-group">');
            const label = $(`<label for="edit-${field.id}">${field.label}</label>`);
            const input = $(`<input type="${field.type}" id="edit-${field.id}" name="${field.name}" 
                            class="form-input" placeholder="${field.placeholder || ''}" 
                            autocomplete="${field.type === 'password' ? 'new-password' : 'new-text'}">`);
            
            let currentValue = this.getFieldValue(field, originalService);
            input.val(currentValue);
            
            formGroup.append(label);
            
            if (field.type === 'password') {
                const passwordField = $('<div class="password-field">');
                const toggleBtn = $('<button type="button" class="password-toggle"><i class="fas fa-eye"></i></button>');
                
                toggleBtn.on('click', function() {
                    const icon = $(this).find('i');
                    if (input.attr('type') === 'password') {
                        input.attr('type', 'text');
                        icon.removeClass('fa-eye').addClass('fa-eye-slash');
                    } else {
                        input.attr('type', 'password');
                        icon.removeClass('fa-eye-slash').addClass('fa-eye');
                    }
                });
                
                passwordField.append(input, toggleBtn);
                formGroup.append(passwordField);
            } else {
                formGroup.append(input);
            }
            
            return formGroup;
        }

        saveEditedService() {
            const serviceId = this.dom.editModal.attr('data-editing-service-id');
            const service = this.data.services.find(s => s.id == serviceId);
            
            this.clearEditModalError();
            
            const editServiceData = {};
            this.dom.editModalBody.find('input').each((i, el) => {
                editServiceData[el.name] = el.value;
            });
            
            this.showEditModalLoading('Updating service...');
            
            const originalService = this.getOriginalServiceData(service.name);
            
            this.apiCall('editMyAiService', {
                '_modelClass': originalService.modelClass,
                '_myAiServiceName': service.name,
                '_editMyAiServiceData': JSON.stringify(editServiceData)
            }).then(response => {
                if (response?.success) {
                    this.showStatus(`"${service.name}" updated successfully`, 'success');
                    this.hideEditServiceModal();
                    this.loadMyAiServices();
                } else {
                    this.showEditModalError('Failed to update service');
                }
            }).catch(() => {
                this.showEditModalError('Failed to update service');
            });
        }

        deleteService(id) {
            if (!confirm('Are you sure you want to delete this service?')) return;
            
            const service = this.data.services.find(s => s.id == id);
            if (service) {
                this.apiCall('revokeMyAiService', { '_myAiServiceName': service.name })
                    .then(response => {
                        if (response?.success) {
                            this.showStatus('Service deleted successfully', 'success');
                            if (response.myAiServices?.properties) {
                                this.updateServicesFromResponse(response.myAiServices.properties.myAiServices);
                            } else {
                                this.data.services = [];
                                this.renderServices();
                                this.updateServiceSelect();
                            }
                        } else {
                            this.showStatus('Failed to delete service', 'error');
                        }
                    })
                    .catch(() => this.showStatus('Failed to delete service', 'error'));
            }
        }

        // [4] PII Masking Rules
        async loadPiiMaskingRules() {
            const response = await this.apiCall('loadPiiMaskingRules');
            if (response?.success && response.piiMaskingRules?.properties) {
                this.data.piiMaskingRules = response.piiMaskingRules.properties.myPiiMaskingRules || [];
                this.renderPiiRules();
            } else {
                this.data.piiMaskingRules = [];
                this.renderPiiRules();
            }
            return true;
        }

        renderPiiRules() {
            // Clear existing rules
            const existingRules = this.dom.piiRulesList.find('.pii-rule-item');
            existingRules.remove();
            
            if (this.data.piiMaskingRules.length === 0) {
                this.dom.piiRulesEmptyState.show();
                return;
            }
            
            this.dom.piiRulesEmptyState.hide();
            
            this.data.piiMaskingRules.forEach((rule, index) => {
                const ruleItem = $(`
                    <div class="pii-rule-item">
                        <div class="pii-rule-text">${this.escapeHtml(rule)}</div>
                        <div class="pii-rule-actions">
                            <button type="button" class="pii-rule-remove" onclick="centralConfig.removePiiRule('${this.escapeHtml(rule)}')">
                                <i class="fas fa-trash"></i> Remove
                            </button>
                        </div>
                    </div>
                `);
                this.dom.piiRulesList.append(ruleItem);
            });
        }

        addPiiRule() {
            const newRule = this.dom.piiRuleInput.val().trim();
            
            // Clear previous error states
            this.clearPiiRuleError();
            
            if (!newRule) {
                this.showPiiRuleError('Please enter a PII masking rule');
                return;
            }
            
            // Check if rule already exists
            if (this.data.piiMaskingRules.includes(newRule)) {
                this.showPiiRuleError('This PII masking rule already exists');
                return;
            }
            
            // Show loading state
            this.dom.addPiiRuleBtn.prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i> Adding...');
            
            this.apiCall('addPiiMaskingRule', { '_piiRule': newRule })
                .then(response => {
                    if (response?.success) {
                        this.showStatus('PII masking rule added successfully!', 'success');
                        this.dom.piiRuleInput.val('');
                        
                        if (response.piiMaskingRules?.properties) {
                            this.data.piiMaskingRules = response.piiMaskingRules.properties.myPiiMaskingRules || [];
                            this.renderPiiRules();
                            
                            // Add highlight animation to the new rule
                            setTimeout(() => {
                                const newRuleItem = this.dom.piiRulesList.find('.pii-rule-item').last();
                                newRuleItem.addClass('newly-added');
                                setTimeout(() => newRuleItem.removeClass('newly-added'), 2000);
                            }, 100);
                        }
                    } else {
                        this.showPiiRuleError(response.error || 'Failed to add PII masking rule');
                    }
                })
                .catch(() => {
                    this.showPiiRuleError('Failed to add PII masking rule');
                })
                .finally(() => {
                    this.dom.addPiiRuleBtn.prop('disabled', false).html('<i class="fas fa-plus"></i> Add Rule');
                });
        }

        removePiiRule(rule) {
            if (!confirm(`Are you sure you want to remove the PII masking rule: "${rule}"?`)) {
                return;
            }
            
            this.apiCall('removePiiMaskingRule', { '_piiRule': rule })
                .then(response => {
                    if (response?.success) {
                        this.showStatus('PII masking rule removed successfully', 'success');
                        
                        if (response.piiMaskingRules?.properties) {
                            this.data.piiMaskingRules = response.piiMaskingRules.properties.myPiiMaskingRules || [];
                            this.renderPiiRules();
                        }
                    } else {
                        this.showStatus('Failed to remove PII masking rule', 'error');
                    }
                })
                .catch(() => {
                    this.showStatus('Failed to remove PII masking rule', 'error');
                });
        }

        showPiiRuleError(message) {
            this.dom.piiRuleInput.addClass('pii-rule-input-error');
            
            // Remove existing error message
            this.dom.piiRuleInput.siblings('.pii-rule-error-message').remove();
            
            // Add error message
            const errorMsg = $(`<div class="pii-rule-error-message show">${message}</div>`);
            this.dom.piiRuleInput.after(errorMsg);
            
            // Auto-hide after 3 seconds
            setTimeout(() => {
                errorMsg.removeClass('show');
                setTimeout(() => errorMsg.remove(), 300);
            }, 3000);
        }

        clearPiiRuleError() {
            this.dom.piiRuleInput.removeClass('pii-rule-input-error');
            this.dom.piiRuleInput.siblings('.pii-rule-error-message').removeClass('show');
        }

        // [4] UI INTERACTION & UPDATE
        // [4.1] Tab Management
        switchTab(tabName) {
            const tabButton = this.dom.tabButtons.filter(`[data-tab="${tabName}"]`);
            if (tabButton.hasClass('disabled')) {
                if (tabName === 'ai-services' || tabName === 'pii-masking') {
                    this.showStatus('Please configure and save API server settings first', 'error');
                }
                return;
            }
            
            this.dom.tabButtons.removeClass('active');
            this.dom.tabContents.removeClass('active');
            
            tabButton.addClass('active');
            $(`#${tabName}-tab`).addClass('active');

            // Load PII rules when switching to PII masking tab
            if (tabName === 'pii-masking' && this.data.piiMaskingRules.length === 0) {
                this.loadPiiMaskingRules();
            }
        }

        async enableAiServicesTab() {
            this.dom.aiServicesTab.removeClass('disabled');
            this.dom.piiMaskingTab.removeClass('disabled');
            
            const statusDot = this.dom.apiServerTab.find('.status-dot');
            if (!statusDot.length) {
                this.dom.apiServerTab.append('<span class="status-dot">');
            }
            this.dom.apiServerTab.find('.status-dot').addClass('configured').removeClass('warning');
            
            await this.loadAvailableModels();
            await this.loadMyAiServices();
            this.loadCurrentAiService();
        }

        disableAiServicesTab() {
            this.dom.aiServicesTab.addClass('disabled');
            this.dom.piiMaskingTab.addClass('disabled');
            
            const statusDot = this.dom.apiServerTab.find('.status-dot');
            if (!statusDot.length) {
                this.dom.apiServerTab.append('<span class="status-dot">');
            }
            this.dom.apiServerTab.find('.status-dot').addClass('warning').removeClass('configured');
        }

        // [4.2] Modal Management
        showAddServiceModal() {
            this.dom.addModal.addClass('show');
            this.showModalLoading('Loading available AI services...');
            this.generateDynamicServiceForm();
        }

        hideAddServiceModal() {
            this.dom.addModal.removeClass('show');
            this.clearServiceForm();
        }

        hideEditServiceModal() {
            this.dom.editModal.removeClass('show');
            this.clearEditServiceForm();
        }

        // [4.3] Status & Loading States
        showConfigLoading() {
            this.dom.status.text('Validating and saving configuration...').attr('class', 'config-status show').css('color', '#0c5460');
            this.dom.saveConfigBtn.prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i> Saving...');
        }

        showStatus(message, type) {
            this.dom.status.text(message).attr('class', `config-status ${type} show`);
            this.dom.saveConfigBtn.prop('disabled', false).html('<i class="fas fa-save"></i> Save Configuration');
            
            const timeout = type === 'info' ? 5000 : 3000;
            setTimeout(() => this.dom.status.removeClass('show'), timeout);
        }

        showModalLoading(message = 'Loading...') {
            this.dom.addModalBody.html(`
                <div class="loading-state" id="modal-loading">
                    <i class="fas fa-spinner fa-spin"></i>
                    <p>${message}</p>
                </div>
            `);
        }

        hideModalLoading() {
            $('#modal-loading').hide();
        }

        clearModalLoading() {
            $('#modal-loading').remove();
        }

        showModalError(message) {
            this.clearModalLoading();
            const existingError = this.dom.addModalBody.find('.modal-error');
            if (existingError.length) existingError.remove();
            
            const errorDiv = $(`
                <div class="modal-error">
                    <div style="background-color: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; 
                                border-radius: 6px; padding: 12px; margin-bottom: 20px; font-size: 14px;">
                        <i class="fas fa-exclamation-triangle" style="margin-right: 8px;"></i>
                        ${message}
                    </div>
                </div>
            `);
            
            this.dom.addModalBody.prepend(errorDiv);
        }

        clearModalError() {
            this.dom.addModalBody.find('.modal-error').remove();
        }

        showEditModalLoading(message = 'Loading...') {
            this.dom.editModalBody.html(`
                <div class="loading-state" id="edit-modal-loading">
                    <i class="fas fa-spinner fa-spin"></i>
                    <p>${message}</p>
                </div>
            `);
        }

        hideEditModalLoading() {
            $('#edit-modal-loading').hide();
        }

        showEditModalError(message) {
            $('#edit-modal-loading').remove();
            const existingError = this.dom.editModalBody.find('.modal-error');
            if (existingError.length) existingError.remove();
            
            const errorDiv = $(`
                <div class="modal-error">
                    <div style="background-color: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; 
                                border-radius: 6px; padding: 12px; margin-bottom: 20px; font-size: 14px;">
                        <i class="fas fa-exclamation-triangle" style="margin-right: 8px;"></i>
                        ${message}
                    </div>
                </div>
            `);
            
            this.dom.editModalBody.prepend(errorDiv);
        }

        clearEditModalError() {
            this.dom.editModalBody.find('.modal-error').remove();
        }

        clearEditServiceForm() {
            this.dom.editModalBody.empty();
            this.clearEditModalError();
        }

        // [5] UTILITY METHODS
        // [5.1] Data Processing
        updateServicesFromResponse(servicesArray) {
            this.data.services = [];
            this.data.originalServicesData = servicesArray;
            
            servicesArray.forEach((service, index) => {
                const serviceData = {
                    id: index + 1,
                    name: service.aiServiceName,
                    modelClass: service.modelClass,
                    modelName: service.modelClass === 'Custom' ? service.customModelName : service.modelName,
                    active: service.using || false
                };
                this.data.services.push(serviceData);
                
                if (service.using) {
                    this.data.currentService = serviceData.id;
                }
            });
            
            this.renderServices();
            this.updateServiceSelect();
        }

        renderServices() {
            const existingServices = this.dom.servicesList.find('.service-item');
            existingServices.remove();
            
            if (this.data.services.length === 0) {
                this.dom.emptyState.show();
                return;
            }
            
            this.dom.emptyState.hide();
            
            this.data.services.forEach(service => {
                const serviceItem = $(`
                    <div class="service-item ${service.active ? 'active' : ''}">
                        <div class="service-info">
                            <div class="service-name">${this.escapeHtml(service.name)}</div>
                            <div class="service-details">
                                <span>Model: ${this.escapeHtml(service.modelName)}</span>
                                <span class="status-badge ${service.active ? 'active' : 'inactive'}">
                                    ${service.active ? 'Active' : 'Inactive'}
                                </span>
                            </div>
                        </div>
                        <div class="service-actions">
                            <button type="button" class="btn-icon" title="Edit" onclick="centralConfig.editService(${service.id})">
                                <i class="fas fa-edit"></i>
                            </button>
                            <button type="button" class="btn-icon" title="Delete" onclick="centralConfig.deleteService(${service.id})">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                    </div>
                `);
                this.dom.servicesList.append(serviceItem);
            });
        }

        updateServiceSelect() {
            this.dom.currentServiceSelect.html('<option value="">Select a service...</option>');
            
            this.data.services.forEach(service => {
                const option = $(`<option value="${service.id}">${service.name}</option>`);
                if (service.id === this.data.currentService) option.prop('selected', true);
                this.dom.currentServiceSelect.append(option);
            });
        }

        handleCurrentServiceChange(value) {
            const saveBtn = $('#save-current-service');
            if (value) {
                if (!saveBtn.length) {
                    const saveButton = $(`
                        <button type="button" id="save-current-service" class="btn btn-primary btn-sm" style="margin-top: 8px;">
                            <i class="fas fa-save"></i> Save Selection
                        </button>
                    `);
                    saveButton.on('click', () => this.saveCurrentAiService());
                    this.dom.currentServiceSelect.parent().append(saveButton);
                }
            } else {
                saveBtn.remove();
            }
        }

        saveCurrentAiService() {
            const currentService = this.dom.currentServiceSelect.val();
            if (!currentService) {
                this.showStatus('Please select a service first', 'error');
                return;
            }
            
            const service = this.data.services.find(s => s.id == currentService);
            if (!service) {
                this.showStatus('Service not found', 'error');
                return;
            }
            
            this.apiCall('saveCurrentAiService', {
                '_currentAiService': service.name,
                '_modelClass': service.modelClass,
                '_modelName': service.modelName
            }).then(response => {
                if (response?.success) {
                    this.showStatus('Current AI Service saved successfully!', 'success');
                    
                    if (response.myAiServices?.properties) {
                        this.updateServicesFromResponse(response.myAiServices.properties.myAiServices);
                    } else {
                        this.data.services.forEach(s => s.active = false);
                        service.active = true;
                        this.data.currentService = currentService;
                        this.renderServices();
                    }
                    
                    $('#save-current-service').remove();
                } else {
                    this.showStatus('Failed to save current service', 'error');
                }
            }).catch(() => {
                this.showStatus('Failed to save current service', 'error');
            });
        }

        // [5.2] Form Helpers
        clearServiceForm() {
            this.dom.addModalBody.find('input').val('');
            this.dom.addModalBody.find('select').prop('selectedIndex', 0);
            this.clearModalError();
        }

        getAllFormFields() {
            const allFields = [];
            this.data.newAiServiceFormFields.fields?.forEach(field => {
                if (!field.children) {
                    allFields.push(field);
                } else {
                    field.children.forEach(child => allFields.push(child));
                }
            });
            return allFields;
        }

        getOriginalServiceData(serviceName) {
            return this.data.originalServicesData?.find(s => s.aiServiceName === serviceName);
        }

        getFieldsForModelClass(modelClass) {
            if (modelClass === "Custom") return ['customLlmUrl', 'customModelName', 'apiKey'];
            if (modelClass === "AWS Bedrock") return ['aws_access_key_id', 'aws_secret_access_key', 'aws_session_token'];
            return ['apiKey'];
        }

        getFieldValue(field, originalService) {
            if (originalService.modelClass === "AWS Bedrock" && field.name.startsWith('aws_')) {
                if (field.name === 'aws_session_token') {
                    return originalService.apiKey[field.name] || '';
                } else {
                    const maskedValue = originalService.apiKey[field.name] || '';
                    return maskedValue.startsWith('%%%%') && maskedValue.endsWith('%%%%') 
                        ? maskedValue : maskedValue;
                }
            } else if (field.name === 'apiKey') {
                const maskedValue = originalService.apiKey || '';
                return maskedValue.startsWith('%%%%') && maskedValue.endsWith('%%%%') 
                    ? maskedValue : maskedValue;
            } else {
                return originalService[field.name] || '';
            }
        }

        shouldActivateValidation(field) {
            if (!field.activateValidationScenarios) return false;
            
            return field.activateValidationScenarios.some(scenario => {
                const allShown = scenario.shown.every(elementId => {
                    const element = $(`#${elementId}`);
                    const containerElements = $(`[data-container="${elementId}"]`);
                    
                    if (element.length) {
                        return !element.closest('.form-group').hasClass('hide');
                    } else if (containerElements.length) {
                        return !containerElements.first().hasClass('hide');
                    }
                    return false;
                });
                
                const allHidden = scenario.hidden.every(elementId => {
                    const element = $(`#${elementId}`);
                    const containerElements = $(`[data-container="${elementId}"]`);
                    
                    if (element.length) {
                        return element.closest('.form-group').hasClass('hide');
                    } else if (containerElements.length) {
                        return containerElements.first().hasClass('hide');
                    }
                    return true;
                });
                
                return allShown && allHidden;
            });
        }

        toggleElementVisibility(elementId, hide) {
            const element = $(`#${elementId}`);
            const containerElements = $(`[data-container="${elementId}"]`);
            
            if (element.length) {
                element.closest('.form-group').toggleClass('hide', hide);
            }
            containerElements.toggleClass('hide', hide);
        }

        // [5.3] State Management
        setupPasswordToggle(toggleId, inputId) {
            $(`#${toggleId}`).on('click', function() {
                const input = $(`#${inputId}`);
                const icon = $(this).find('i');
                
                if (input.attr('type') === 'password') {
                    input.attr('type', 'text');
                    icon.removeClass('fa-eye').addClass('fa-eye-slash');
                } else {
                    input.attr('type', 'password');
                    icon.removeClass('fa-eye-slash').addClass('fa-eye');
                }
            });
        }

        escapeHtml(text) {
            return $('<div>').text(text).html();
        }

        async apiCall(action, data = {}) {
            return new Promise((resolve, reject) => {
                $.ajax({
                    type: "POST",
                    url: UI.base + `/web/json/plugin/org.joget.ai.designer.AiDesignerCentralConfig/service`,
                    data: { '_a': action, ...data },
                    dataType: action.includes('validate') ? "text" : "json",
                    success: resolve,
                    error: (xhr, status, error) => {
                        console.error(`Error in ${action}:`, error);
                        reject(error);
                    }
                });
            });
        }
    }

    // [6] JQUERY PLUGIN WRAPPER
    $.fn.centralConfig = function (options) {
        return this.each(function () {
            if (!$.data(this, 'plugin_centralConfig')) {
                const instance = new CentralConfig(this, options);
                $.data(this, 'plugin_centralConfig', instance);
                
                // Expose instance globally for onclick handlers
                window.centralConfig = instance;
            }
        });
    };

})(jQuery);

// Initialize when page loads
$(document).ready(function() {
    $('#config-container').centralConfig({
        debug: true
    });
});