/*!
 * FormInjectorPopUp jQuery Plugin (OOP, Multi-Step)
 * -------------------------------------------
 *
 * TABLE OF CONTENTS
 * [1] CLASS DEFINITION: FormInjectorPopUp
 *   [1.1] Constructor & Initialization
 *   [1.2] DOM Caching
 *   [1.3] Event Binding
 * [2] WEBSOCKET HANDLING
 *   [2.1] Init/Connect WebSocket
 *   [2.2] Send Message on WebSocket
 *   [2.3] Handle Incoming WebSocket Messages
 *   [2.4] Process image file
 * [3] UI INTERACTION & UPDATE
 *   [3.1] Update Connection Status
 *   [3.2] Message List: Add/Scroll/Step Actions
 *   [3.3] Refine Step Input
 *   [3.4] Send/Receive/Typing Controls
 *   [3.5] Hide & Show Input Area
 * [4] CONTENT FORMATTING AND UTILS
 *   [4.1] Step Content Formatting
 *   [4.2] Text Escape
 *   [4.3] Typing Dots Animation
 * [5] JQUERY PLUGIN WRAPPER
 */

(function ($) {

    // [1] CLASS DEFINITION: FormInjectorPopUp
    class FormInjectorPopUp {
        // [1.1] Constructor & Initialization
        constructor(el, options) {
            this.options = options || {};
            this.chatEl = $(el);
            this.sessionId = 'form_session_' + Math.random().toString(36).substr(2, 9);
            this.ws = null;
            this.isConnected = false;
            this.isInitialized = false;
            this.lock = false;
            this.uploadedImageBase64 = null;
            this.uploadedImageFileName = '';
            this.isImageMode = false;

            // [1.2] DOM Caching
            this.cacheDom();

            // [1.3] Event Binding
            this.bindEvents();

            // [2.1] Init/Connect WebSocket on load
            this.initWebSocket();

            // Cleanup on window unload
            $(window).on('beforeunload', () => {
                if (this.ws) this.ws.close();
            });
        }

        // [1.2] Cache all DOM nodes for easy/fast access
        cacheDom() {
            this.dom = {
                chatIcon:       this.chatEl.find('#formChatIcon'),
                chatPopup:      this.chatEl.find('#formChatPopup'),
                sendButton:     this.chatEl.find('#form-chat-send-btn'),
                chatForm:       this.chatEl.find('#form-chat-form'),
                userInput:      this.chatEl.find('#form-user-input'),
                messages:       this.chatEl.find('#form-messages'),
                messagesCon:    this.chatEl.find('.msger-messages'),
                headerTitle:    this.chatEl.find('#form-msger-header-title'),
                uploadBtn:      this.chatEl.find('#form-upload-image-btn'),
                fileInput:      this.chatEl.find('#form-file-input'),
                imageContainer: this.chatEl.find('#form-uploaded-image-container'),
                imagePreview:   this.chatEl.find('#form-image-preview'),
                imageFilename:  this.chatEl.find('#form-image-filename'),
                removeImageBtn: this.chatEl.find('#form-remove-image-btn')
            };
        }

        // [1.3] Wire up all UI events and handlers
        bindEvents() {
            const d = this.dom;

            d.chatIcon.off('click.formInjectorPopUp').on('click.formInjectorPopUp', e => {
                d.chatPopup.toggleClass('show');
                if (d.chatPopup.hasClass('show')) {
                    if (!this.ws || this.ws.readyState === WebSocket.CLOSED) {
                        this.initWebSocket();
                    } else if (this.ws.readyState === WebSocket.OPEN && !this.isInitialized) {
                        this.sendWebSocketMessage({ type: 'init', sessionId: this.sessionId });
                        this.isInitialized = true;
                    }
                    if (!this.isImageMode) {
                        d.userInput.focus();
                    }
                }
            });

            d.chatForm.off('submit.formInjectorPopUp').on('submit.formInjectorPopUp', e => {
                e.preventDefault();
                this.sendMessage();
                return false;
            });

            d.sendButton.off('click.formInjectorPopUp').on('click.formInjectorPopUp', e => {
                e.preventDefault();
                this.sendMessage();
            });

            d.userInput.off('keypress.formInjectorPopUp').on('keypress.formInjectorPopUp', e => {
                if (e.which === 13 && !e.shiftKey) {
                    e.preventDefault();
                    this.sendMessage();
                }
            });

            // Image upload events
            d.uploadBtn.on('click.formInjectorPopUp', e => {
                e.preventDefault();
                if (this.lock) {
                    return;
                }
                d.fileInput.click();
            });

            d.fileInput.on('change.formInjectorPopUp', e => {
                if (e.target.files && e.target.files[0]) {
                    this.handleImageUpload(e.target.files[0]);
                }
            });

            d.removeImageBtn.on('click.formInjectorPopUp', e => {
                e.preventDefault();
                this.removeUploadedImage();
            });
        }

        // [2.1] WebSocket connection lifecycle & reconnect logic
        initWebSocket() {
            try {
                const protocol = window.location.protocol === "https:" ? "wss://" : "ws://";
                const wsUrl = protocol + window.location.host + this.options.contextPath + "/web/socket/plugin/org.joget.ai.designer.feature.FormInjectorPopUp";
                this.ws = new WebSocket(wsUrl);

                this.ws.onopen = () => {
                    this.log('WebSocket connected');
                    this.isConnected = true;
                    this.isInitialized = false;
                    this.updateConnectionStatus(true);
                    this.sendWebSocketMessage({ type: 'init', sessionId: this.sessionId });
                    this.isInitialized = true;
                };
                
                this.ws.onmessage = (event) => {
                    this.log('Received message:', event.data);
                    this.handleWebSocketEventData(event.data);
                };
                
                this.ws.onclose = () => {
                    this.isConnected = false;
                    this.isInitialized = false;
                    this.updateConnectionStatus(false);
                    
                    // Unlock UI if connection closes
                    this.lock = false;
                    this.dom.userInput.prop('disabled', false);
                    this.dom.sendButton.prop('disabled', false).text('Send');
                    this.hideTypingIndicator();
                    
                    setTimeout(() => {
                        if (!this.isConnected) {
                            this.log('Attempting to reconnect...');
                            this.initWebSocket();
                        }
                    }, 3000);
                };
                
                this.ws.onerror = (error) => {
                    this.isConnected = false;
                    this.isInitialized = false;
                    this.updateConnectionStatus(false);
                    
                    // Unlock UI on error
                    this.lock = false;
                    this.dom.userInput.prop('disabled', false);
                    this.dom.sendButton.prop('disabled', false).text('Send');
                    this.hideTypingIndicator();
                };
            } catch (error) {
                this.appendMessage('Connection failed. Please refresh the page.', 'received error');
            }
        }

        // [2.2] Send data/messages through WebSocket
        sendWebSocketMessage(messageObj) {
            if (this.ws && this.ws.readyState === WebSocket.OPEN) {
                this.log('Sending WebSocket message:', JSON.stringify(messageObj));
                this.ws.send(JSON.stringify(messageObj));
                if (messageObj.type === 'init') this.dom.messages?.empty();
                return true;
            }
            this.appendMessage('Connection lost. Trying to reconnect...', 'received error');
            return false;
        }

        // [2.3] WebSocket: Incoming messages & error handling
        handleWebSocketEventData(data) {
            // Always clear waiting when a new message comes in
            this.lock = false;
            this.dom.userInput.prop('disabled', this.lock);
            this.dom.sendButton.prop('disabled', this.lock).text('Send');

            this.hideTypingIndicator();
            try {
                const message = JSON.parse(data);
                this.handleWebSocketMessage(message);
            } catch (e) {
                this.appendMessage(data, 'received');
            }
        }

        // [2.3] WebSocket: Dispatch specific message types
        handleWebSocketMessage(message) {
            switch (message.type) {
                case 'step_1_form_design':
                case 'step_1_form_design_refined':
                    this.appendMessage(this.formatStepContent(message.content, message.step), 'received');
                    if (message.type === 'step_1_form_design_refined') {
                        this.appendMessage('✨ Form has been refined based on your feedback!', 'received success');
                    }
                    this.appendStepActions(message.actions);
                    break;
                case 'step_2_form_inject':
                    this.appendMessage(this.formatStepContent(message.content, message.step), 'received success');
                    if (message.content && message.content.formJson) {
                        this.injectFormIntoBuilder(message.content.formJson);
                    }
                    // Reset form state and show input for new form
                    setTimeout(() => {
                        this.resetFormState();
                        this.showInputArea();
                        this.appendMessage('Form injected successfully! You can design another form or close this chat.', 'received');
                    }, 1000);
                    break;
                case 'chat_response':
                    message.content
                        ? this.appendMessage(message.content, 'received')
                        : this.appendMessage('Error: ' + message.error, 'received error');
                    break;
                case 'error':
                    this.appendMessage('Error: ' + (message.message || 'Unknown error occurred'), 'received error');
                    this.showInputArea();
                    break;
                case 'status':
                    if (message.message) this.log('Status:', message.message);
                    break;
                default:
                    const content = message.content || message.response || message.answer;
                    this.appendMessage(content ? content : JSON.stringify(message), 'received');
            }
        }

        // [2.4] Process image file
        handleImageUpload(file) {
            // Validate file type
            if (!file.type.startsWith('image/')) {
                alert('Please upload an image file.');
                this.dom.fileInput.val('');
                return;
            }

            // Validate file size (2MB limit)
            if (file.size > 2 * 1024 * 1024) {
                alert('File size must be less than 2MB.');
                this.dom.fileInput.val('');
                return;
            }

            this.uploadedImageFileName = file.name;

            const reader = new FileReader();
            reader.onload = (e) => {
                const img = new Image();
                img.onload = () => {
                    const canvas = document.createElement('canvas');
                    const ctx = canvas.getContext('2d');

                    canvas.width = img.width;
                    canvas.height = img.height;
                    ctx.drawImage(img, 0, 0);

                    this.uploadedImageBase64 = canvas.toDataURL('image/jpeg');
                    this.showImagePreview();
                    this.updateInputMode();
                    
                };
                img.src = e.target.result;
            };
            reader.readAsDataURL(file);
            this.dom.fileInput.val(''); // Reset file input
        }

        // [3.1] UI: Indicate connection (header/status)
        updateConnectionStatus(connected) {
            this.dom.headerTitle.html(
                '<i class="fas fa-robot"></i> AI Designer - Form Injector ' +
                `<span class="status-indicator ${connected ? 'connected' : 'disconnected'}"></span>`
            );
        }

        // [3.2] Message List: Add/Scroll/Step Actions
        appendMessage(message, className) {
            if (className.includes('received')) {
                // Create message with avatar for system messages
                const messageWrapper = $('<li class="message-with-avatar"></li>');
                
                // Create avatar
                const avatarContainer = $(`
                    <div class="avatar-container">
                        <div class="avatar">
                            <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 32 32" class="svg-icon">
                                <path d="M 16 3 C 8.832 3 3 8.832 3 16 C 3 23.168 8.832 29 16 29 C 23.168 29 29 23.168 29 16 C 29 8.832 23.168 3 16 3 z M 16 5 C 22.065 5 27 9.935 27 16 C 27 16.402757 26.976005 16.79973 26.933594 17.191406 C 24.411573 17.18943 17.473247 17.233463 14.953125 17.167969 C 11.872125 17.098969 10.418938 16.012953 11.710938 14.501953 C 12.991937 12.978953 14.226031 12.228625 15.207031 11.640625 C 16.199031 11.063625 16.684141 9.8063906 13.869141 10.025391 C 8.3648999 10.399383 6.0625804 13.440264 5.0058594 15.867188 C 5.0780433 9.8637834 9.979814 5 16 5 z M 21.582031 12 C 20.589031 12 18.916609 13.419516 17.474609 14.353516 C 15.732609 15.495516 13.851562 16.095703 13.851562 16.095703 L 23.255859 16.003906 L 25.988281 14.664062 C 25.988281 14.664063 26.184734 14.133547 24.927734 13.810547 C 23.669734 13.475547 21.616797 14.469422 19.966797 15.357422 C 18.316797 16.234422 18.248047 15.806641 18.248047 15.806641 C 18.248047 15.806641 20.831203 13.869719 21.408203 13.511719 C 21.997203 13.154719 23.185203 12.738391 22.158203 12.150391 C 21.973203 12.046391 21.778031 12 21.582031 12 z M 26.091797 20.367188 C 24.398056 24.264986 20.513385 27 16 27 C 15.298844 27 14.614291 26.926814 13.949219 26.800781 C 13.565765 26.168524 13.195705 25.506682 13.037109 25.140625 C 12.703109 24.367625 11.468891 22.56675 14.837891 21.34375 C 17.532234 20.368736 23.588388 20.392244 26.091797 20.367188 z" />
                            </svg>
                        </div>
                    </div>
                `);
                
                // Create message content
                const messageEl = $('<div class="message ' + className + '"></div>');
                if (message && message.includes && message.includes('<')) {
                    messageEl.html(message);
                } else {
                    messageEl.text(message);
                }
                
                messageWrapper.append(avatarContainer).append(messageEl);
                this.dom.messages.append(messageWrapper);
            } else {
                // Regular message for sent messages
                const el = $('<li class="message ' + className + '"></li>');
                if (className.includes('received') && message && message.includes && message.includes('<')) {
                    el.html(message);
                } else {
                    el.text(message);
                }
                this.dom.messages.append(el);
            }
            this.scrollToBottom();
        }

        appendStepActions(actions) {
            if (!actions || actions.length === 0) return;
            
            const actionsHtml = $('<div class="step-actions"></div>');
            actions.forEach(a => {
                $('<button class="step-action-btn ' + a + '"></button>')
                    .text(a.charAt(0).toUpperCase() + a.slice(1))
                    .on('click', () => {
                        this.dom.messages.find('li.message.action').remove();
                        
                        if (a === "refine") {
                            this.showRefineInput();
                            return;
                        }
                        
                        if (a === "restart") {
                            this.resetFormState();
                            this.showInputArea();
                            this.dom.messages.empty();
                            this.appendMessage('Welcome back! Describe the form you\'d like to create or upload an image.', 'received');
                            return;
                        }
                        
                        this.lock = true;
                        this.dom.userInput.prop('disabled', this.lock);
                        this.dom.sendButton.prop('disabled', this.lock);
                        this.showTypingIndicator();
                        
                        this.sendWebSocketMessage({
                            type: "continue_step",
                            sessionId: this.sessionId
                        });
                    })
                    .appendTo(actionsHtml);
            });
            this.dom.messages.append($('<li class="message action"></li>').append(actionsHtml));
            this.scrollToBottom();
        }

        // [3.2] Scroll Chat to Bottom Helper
        scrollToBottom = () => this.dom.messagesCon.scrollTop(this.dom.messagesCon[0].scrollHeight);

        // [3.3] UI: Sending Logic & Typing Indicator
        sendMessage() {
            const d = this.dom;
            const text = d.userInput.val().trim();

            const hasText = text !== '';
            const hasImage = this.isImageMode && (this.uploadedImageBase64 !== null);

            // Validate input
            if (!hasText && !hasImage) {
                alert('Please enter a message or upload an image.');
                return;
            }

            if (!this.isConnected) {
                this.appendMessage('Connection lost. Trying to reconnect...', 'received error');
                return;
            }

            if (this.lock) {
                console.log('Blocked by lock');
                return;
            }
            
            this.lock = true;

            // Clear text input only if not in image mode
            if (!this.isImageMode) {
                d.userInput.val('');
            }
            d.userInput.prop('disabled', this.lock);
            d.sendButton.prop('disabled', this.lock).text('Sending...');

            // Show appropriate sent message
            if (hasImage) {
                this.appendImageMessage();
            } else {
                this.appendMessage(text, 'sent');
            }

            this.showTypingIndicator();
            this.hideInputArea();

            // Prepare image data for WebSocket (extract base64 without data URL)
            let imageBase64Clean = null;
            if (hasImage) {
                imageBase64Clean = this.uploadedImageBase64;
                if (imageBase64Clean.includes("base64,")) {
                    imageBase64Clean = imageBase64Clean.split("base64,")[1];
                }
            }

            const messageData = {
                type: 'chat_message',
                message: hasImage ? '' : text,
                hasImage: hasImage,
                imageBase64: imageBase64Clean,
                imageFileName: this.uploadedImageFileName || 'image.jpg',
                sessionId: this.sessionId,
                timestamp: new Date().toISOString()
            };

            if (!this.sendWebSocketMessage(messageData)) {
                this.hideTypingIndicator();
                this.appendMessage('Failed to send message. Please try again.', 'received error');
                this.showInputArea();
            } else {
                console.log('Message sent successfully');
            }
        }

        // [3.3] Refine Step Input
        showRefineInput() {
            const refineWrapper = $('<li class="message-with-avatar"></li>');
            
            // Create avatar for refine input
            const avatarContainer = $(`
                <div class="avatar-container">
                    <div class="avatar">
                        <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 32 32" class="svg-icon">
                            <path d="M 16 3 C 8.832 3 3 8.832 3 16 C 3 23.168 8.832 29 16 29 C 23.168 29 29 23.168 29 16 C 29 8.832 23.168 3 16 3 z M 16 5 C 22.065 5 27 9.935 27 16 C 27 16.402757 26.976005 16.79973 26.933594 17.191406 C 24.411573 17.18943 17.473247 17.233463 14.953125 17.167969 C 11.872125 17.098969 10.418938 16.012953 11.710938 14.501953 C 12.991937 12.978953 14.226031 12.228625 15.207031 11.640625 C 16.199031 11.063625 16.684141 9.8063906 13.869141 10.025391 C 8.3648999 10.399383 6.0625804 13.440264 5.0058594 15.867188 C 5.0780433 9.8637834 9.979814 5 16 5 z M 21.582031 12 C 20.589031 12 18.916609 13.419516 17.474609 14.353516 C 15.732609 15.495516 13.851562 16.095703 13.851562 16.095703 L 23.255859 16.003906 L 25.988281 14.664062 C 25.988281 14.664063 26.184734 14.133547 24.927734 13.810547 C 23.669734 13.475547 21.616797 14.469422 19.966797 15.357422 C 18.316797 16.234422 18.248047 15.806641 18.248047 15.806641 C 18.248047 15.806641 20.831203 13.869719 21.408203 13.511719 C 21.997203 13.154719 23.185203 12.738391 22.158203 12.150391 C 21.973203 12.046391 21.778031 12 21.582031 12 z M 26.091797 20.367188 C 24.398056 24.264986 20.513385 27 16 27 C 15.298844 27 14.614291 26.926814 13.949219 26.800781 C 13.565765 26.168524 13.195705 25.506682 13.037109 25.140625 C 12.703109 24.367625 11.468891 22.56675 14.837891 21.34375 C 17.532234 20.368736 23.588388 20.392244 26.091797 20.367188 z" />
                        </svg>
                    </div>
                </div>
            `);
            
            // Create refine input form
            const refineInputHtml = $(`
                <div class="message received refine-input">
                    <div class="refine-title">🔧 Refine the form design</div>
                    <div class="refine-description">What changes would you like to make?</div>
                    <div class="refine-form">
                        <textarea id="refine-textarea" placeholder="e.g., Add an employee email field, change leave types, etc." 
                                rows="3" class="refine-input-field"></textarea>
                        <div class="refine-buttons">
                            <button id="refine-submit-btn" class="refine-btn primary">Apply Changes</button>
                            <button id="refine-cancel-btn" class="refine-btn secondary">Cancel</button>
                        </div>
                    </div>
                </div>
            `);
            
            refineWrapper.append(avatarContainer).append(refineInputHtml);
            this.dom.messages.append(refineWrapper);
            this.scrollToBottom();
            
            // Focus on textarea
            setTimeout(() => {
                $('#refine-textarea').focus();
            }, 100);
            
            // Handle refine submit
            $('#refine-submit-btn').on('click', () => {
                const refineText = $('#refine-textarea').val().trim();
                if (!refineText) {
                    alert('Please enter your refinement instructions.');
                    return;
                }
                
                // Remove the refine input
                refineWrapper.remove();
                
                // Show user's refine request
                this.appendMessage(`🔧 Refine request: ${refineText}`, 'sent');
                
                // Lock UI and show typing
                this.lock = true;
                this.dom.userInput.prop('disabled', this.lock);
                this.dom.sendButton.prop('disabled', this.lock);
                this.showTypingIndicator();
                
                // Send refine request
                this.sendWebSocketMessage({
                    type: 'refine_form',
                    message: refineText,
                    sessionId: this.sessionId
                });
            });
            
            // Handle refine cancel
            $('#refine-cancel-btn').on('click', () => {
                refineWrapper.remove();
                // Re-show the action buttons
                this.appendStepActions(['continue', 'refine', 'restart']);
            });
            
            // Handle Enter key (with Shift+Enter for newline)
            $('#refine-textarea').on('keypress', (e) => {
                if (e.which === 13 && !e.shiftKey) {
                    e.preventDefault();
                    $('#refine-submit-btn').click();
                }
            });
        }

        // [3.4] Typing Loader
        showTypingIndicator() {
            const typingWrapper = $('<li class="message-with-avatar"></li>');
            
            // Create avatar for typing indicator
            const avatarContainer = $(`
                <div class="avatar-container">
                    <div class="avatar">
                        <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 32 32" class="svg-icon">
                            <path d="M 16 3 C 8.832 3 3 8.832 3 16 C 3 23.168 8.832 29 16 29 C 23.168 29 29 23.168 29 16 C 29 8.832 23.168 3 16 3 z M 16 5 C 22.065 5 27 9.935 27 16 C 27 16.402757 26.976005 16.79973 26.933594 17.191406 C 24.411573 17.18943 17.473247 17.233463 14.953125 17.167969 C 11.872125 17.098969 10.418938 16.012953 11.710938 14.501953 C 12.991937 12.978953 14.226031 12.228625 15.207031 11.640625 C 16.199031 11.063625 16.684141 9.8063906 13.869141 10.025391 C 8.3648999 10.399383 6.0625804 13.440264 5.0058594 15.867188 C 5.0780433 9.8637834 9.979814 5 16 5 z M 21.582031 12 C 20.589031 12 18.916609 13.419516 17.474609 14.353516 C 15.732609 15.495516 13.851562 16.095703 13.851562 16.095703 L 23.255859 16.003906 L 25.988281 14.664062 C 25.988281 14.664063 26.184734 14.133547 24.927734 13.810547 C 23.669734 13.475547 21.616797 14.469422 19.966797 15.357422 C 18.316797 16.234422 18.248047 15.806641 18.248047 15.806641 C 18.248047 15.806641 20.831203 13.869719 21.408203 13.511719 C 21.997203 13.154719 23.185203 12.738391 22.158203 12.150391 C 21.973203 12.046391 21.778031 12 21.582031 12 z M 26.091797 20.367188 C 24.398056 24.264986 20.513385 27 16 27 C 15.298844 27 14.614291 26.926814 13.949219 26.800781 C 13.565765 26.168524 13.195705 25.506682 13.037109 25.140625 C 12.703109 24.367625 11.468891 22.56675 14.837891 21.34375 C 17.532234 20.368736 23.588388 20.392244 26.091797 20.367188 z" />
                        </svg>
                    </div>
                </div>
            `);
            
            // Create typing message
            const typingMessage = $('<div class="message received typing" id="typing-indicator">AI is designing form<span class="dots">...</span></div>');
            
            typingWrapper.append(avatarContainer).append(typingMessage);
            this.dom.messages.append(typingWrapper);
            this.scrollToBottom();
            this.animateTypingDots();
        }

        // [3.5] Hide & Show Input Area
        hideInputArea() {
            this.dom.chatForm.hide();
            this.log('Input area hidden');
        }

        showInputArea() {
            this.dom.chatForm.show();
            if (!this.isImageMode) {
                this.dom.userInput.focus();
            }
            this.log('Input area shown');
        }

        // method to reset form state
        resetFormState() {
            this.removeUploadedImage();
            this.dom.userInput.val('');
            this.updateInputMode();
        }

        // Add method to show sent image in chat
        appendImageMessage() {
            const imageWrapper = $('<li class="message sent image-message"></li>');
            const imageContent = $(`
                <div class="sent-image-container">
                    <img src="${this.uploadedImageBase64}" alt="${this.uploadedImageFileName}" class="sent-image" />
                    <div class="sent-image-filename">${this.uploadedImageFileName}</div>
                </div>
            `);
            
            imageWrapper.append(imageContent);
            this.dom.messages.append(imageWrapper);
            this.scrollToBottom();
        }

        showImagePreview() {
            this.dom.imagePreview.attr('src', this.uploadedImageBase64);
            this.dom.imageFilename.text(this.uploadedImageFileName);
            this.dom.imageContainer.show();
            this.isImageMode = true;
        }

        removeUploadedImage() {
            this.uploadedImageBase64 = null;
            this.uploadedImageFileName = '';
            this.isImageMode = false;
            this.dom.imageContainer.hide();
            this.dom.imagePreview.attr('src', '');
            this.dom.imageFilename.text('');
            this.updateInputMode();
        }

        updateInputMode() {
            if (this.isImageMode) {
                // Hide text input when image is uploaded
                this.dom.userInput.addClass('hidden');
                this.dom.uploadBtn.attr('title', 'Replace Image');
            } else {
                // Show text input when no image
                this.dom.userInput.removeClass('hidden');
                this.dom.uploadBtn.attr('title', 'Upload Image');
            }
        }

        hideTypingIndicator = () => this.dom.messages.find('#typing-indicator').closest('li').remove(); 
        
        // [4.3] Typing Animation Dots (UI flair)
        animateTypingDots() {
            const dots = this.dom.messages.find('#typing-indicator .dots');
            if (dots.length) {
                let dotCount = 0;
                const interval = setInterval(() => {
                    if (!this.dom.messages.find('#typing-indicator').length) return clearInterval(interval);
                    dotCount = (dotCount % 3) + 1;
                    dots.text('.'.repeat(dotCount));
                }, 500);
            }
        }

        // [4.1] Format main content of each step (for appendMessage)
        formatStepContent(content, step) {
            if (step == 1) {
                // Format designed_forms for display
                if (content && typeof content === "object") {
                    let html = '<div class="form-structure"><h4>📋 Form Design Preview</h4>';
                    
                    Object.keys(content).forEach(formName => {
                        html += `<div class="form-section"><h5>${FormInjectorPopUp.escapeHtml(formName)}</h5>`;
                        const formSections = content[formName];
                        
                        Object.keys(formSections).forEach(sectionName => {
                            html += `<div style="margin-left: 16px;"><strong>${FormInjectorPopUp.escapeHtml(sectionName)}:</strong><br>`;
                            const fields = formSections[sectionName];
                            
                            Object.keys(fields).forEach(fieldName => {
                                const fieldType = fields[fieldName];
                                html += `<div class="form-field"><span class="field-name">${FormInjectorPopUp.escapeHtml(fieldName)}</span>: <span class="field-type">${FormInjectorPopUp.escapeHtml(fieldType)}</span></div>`;
                            });
                            html += '</div>';
                        });
                        html += '</div>';
                    });
                    html += '</div>';
                    return html;
                }
                return '<pre>' + FormInjectorPopUp.escapeHtml(JSON.stringify(content, null, 2)) + '</pre>';
            } else if (step == 2) {
                // Form injection success message
                if (content && content.successMessage) {
                    return '<div class="success-message">✅ ' + FormInjectorPopUp.escapeHtml(content.successMessage) + '</div>';
                }
                return FormInjectorPopUp.escapeHtml(JSON.stringify(content));
            }
            return FormInjectorPopUp.escapeHtml(JSON.stringify(content));
        }

        // Inject form into CustomBuilder
        injectFormIntoBuilder(formJson) {
            try {
                // Check if CustomBuilder is available
                if (typeof CustomBuilder === 'undefined') {
                    throw new Error('CustomBuilder is not available. Make sure you are in the form builder page.');
                }
                
                // Inject the form using CustomBuilder API
                if (typeof CustomBuilder.loadJson === 'function') {
                    let newFormJson = JSON.stringify(formJson);
                    CustomBuilder.loadJson(newFormJson, true);
                    this.log('Form injected successfully using CustomBuilder.loadJson()', formJson);
                } else {
                    throw new Error('CustomBuilder.loadJson() method is not available.');
                }
                
            } catch (error) {
                console.error('Failed to inject form using CustomBuilder:', error);
                this.appendMessage('Could not inject form automatically: ' + error.message, 'received error');
            }
        }

        // [4.2] Escape HTML to prevent XSS
        static escapeHtml = (str) => $('<div/>').text(str).html();

        log = (...args) => { if (this.options.debug) console.log(...args); }
        
    }

    // [5] jQuery Plugin Wrapper (singleton for each root)
    $.fn.formInjectorPopUp = function (options) {
        return this.each(function () {
            if (!$.data(this, 'plugin_formInjectorPopUp')) {
                $.data(this, 'plugin_formInjectorPopUp', new FormInjectorPopUp(this, options));
            }
        });
    };

})(jQuery);