ApiConnectorBuilder = {
    oasCache : {},
    
    /*
     * Intialize the builder, called from CustomBuilder.initBuilder
     */
    initBuilder: function (callback) {
        $("#design-btn").before('<button class="btn btn-light" id="oas-btn" type="button" data-toggle="button" aria-pressed="true" data-cbuilder-view="oasSelector" data-cbuilder-action="switchView" title="'+get_cbuilder_msg('apiConnectorBuilder.OAS')+'"><i class="las la-file-code"></i> <span>'+get_cbuilder_msg('apiConnectorBuilder.OAS')+'</span> </button>');
        
        ApiConnectorBuilder.initDefaultComponent();
        
        CustomBuilder.Builder.init({
            "enableViewport": false,
            callbacks : {
                "renderElement" : "ApiConnectorBuilder.renderElement"
            }
        }, function() {
            var deferreds = [];
            
            CustomBuilder.Builder.setHead('<link data-datalist-style href="' + CustomBuilder.contextPath + '/plugin/org.joget.connector.ApiConnectorBuilder/static/api_connector_builder.css" rel="stylesheet" />');
            if (CustomBuilder.systemTheme === undefined) {
                CustomBuilder.systemTheme = $('body').attr("builder-theme");
            }
            if (CustomBuilder.systemTheme === 'dark') {
                CustomBuilder.Builder.setHead('<link data-userview-style href="' + CustomBuilder.contextPath + '/css/darkTheme.css" rel="stylesheet" />');
            }
            
            $.when.apply($, deferreds).then(function() {
                if (callback) {
                    callback();
                }
            });
        });
    },
    
    /*
     * Create a component for root element
     */
    initDefaultComponent : function() {
        CustomBuilder.initPaletteElement("", "org.joget.connector.ApiConnectorBuilder", "", "", "", "", false, "", {
            builderTemplate: {
                'draggable' : false,
                'movable' : false,
                'deletable' : false,
                'copyable' : false,
                'navigable' : false,
                'supportProperties' : false,
                'supportStyle' : false,
                'uneditable' : true
            }
        });
    },
    
    /*
     * Render the element to canvas
     */
    renderElement : function(element, elementObj, component, callback) {
        if (elementObj.className === "org.joget.connector.ApiConnectorBuilder") {
            $(element).addClass("apisContainer");
            $(element).attr("data-cbuilder-elements", "");
            $(element).attr("data-cbuilder-droparea-msg", get_cbuilder_msg('apiConnectorBuilder.dragApiToHere'));
            $(element).data("data", elementObj);
            
            callback(element);
        } else {
            ApiConnectorBuilder.renderPathElement(element, elementObj, component, callback);
        }
    },
    
    /*
     * Render path element to canvas
     */
    renderPathElement : function(element, elementObj, component, callback) {
        $(element).addClass("path_container").addClass(component.method);
        
        var summary = component.pathData.summary;
        if (elementObj.properties.name !== undefined && elementObj.properties.name !== "") {
            summary = elementObj.properties.name;
        }
        
        $(element).html('<h3 class="path"><span class="path_method">'+component.method+'</span> <span class="path_info">'+component.path+'</span> <span class="path_summary">'+summary+'</span></h3>');
        
        callback(element);
    },
    
    /*
     * Show OAS page
     */
    showOasSelector : function(){
        if (!$("#oas-btn").hasClass("active-view")) {
            setTimeout(function(){
                $("#oas-btn").trigger("click");
            }, 1);
        }
    },
    
    /*
     * Load and render data, called from CustomBuilder.loadJson
     */
    load: function (data) {
        var self = CustomBuilder.Builder;
        
        if (data.properties.oas === undefined || data.properties.oas === "") {
            ApiConnectorBuilder.showOasSelector();
            return;
        }
        
        delete ApiConnectorBuilder.oas;
        
        var deferreds = [];
        
        ApiConnectorBuilder.retrieveOAS(data.properties.oas, deferreds, true);  
        
        $.when.apply($, deferreds).then(function() {
            CustomBuilder.Builder.load(data);
        });    
    },
    
    /*
     * A callback method from CustomBuilder.switchView to render OAS selector properties editor
     */
    oasSelectorViewInit: function(view) {
        $(view).html("");
        
        var options = {
            appPath: CustomBuilder.appPath,
            contextPath: CustomBuilder.contextPath,
            propertiesDefinition : ApiConnectorBuilder.getOasPropertiesDefinition(),
            propertyValues : CustomBuilder.data.properties,
            showCancelButton:false,
            closeAfterSaved : false,
            changeCheckIgnoreUndefined: true,
            autoSave: true,
            saveCallback: function(container, properties) {
                delete properties.downloadOas;
                delete properties.config;
                
                //if oas document changed, start new
                if (CustomBuilder.data.properties.oas !== properties.oas) {
                    CustomBuilder.data.elements = [];
                    CustomBuilder.data.components = {};
                }
                
                CustomBuilder.data.properties = $.extend(CustomBuilder.data.properties, properties);
                
                //copy security scehema
                ApiConnectorBuilder.updateOasSchema();
                
                //delete oas from cache
                delete ApiConnectorBuilder.oasCache[CustomBuilder.appPath + "/resources/" + properties.oas];
                
                CustomBuilder.update();
                ApiConnectorBuilder.load(CustomBuilder.data);
            }
        };
        $("body").addClass("stop-scrolling");
        
        $(view).off("builder-view-show");
        $(view).on("builder-view-show", function(){
            $(view).propertyEditor(options);
            
            //render download button
            setTimeout(function(){
                $(view).find('[property-name="downloadOAS"] .property-input').prepend('<button id="downloadOAS" class="btn btn-sm btn-secondary"><i class="fas fa-download"></i> '+get_cbuilder_msg('apiConnectorBuilder.download')+'</button>');
                
                $(view).find('[property-name="downloadOAS"] .property-input #downloadOAS').off("click").on("click", function() {
                    var oasDoc = $(view).find('[property-name="oas"] .property-input input').val();
                    
                    var url = CustomBuilder.contextPath + "/web/app" + CustomBuilder.appPath + "/resources/" + oasDoc;

                    var link = document.createElement('a');
                    link.href = url;
                    link.download = oasDoc;

                    document.body.appendChild(link);
                    link.click();

                    // Clean up: Remove the link and revoke the URL
                    document.body.removeChild(link);
                });
            }, 100);
        });
    },
    
    /*
     * Get binder properties definition for data view
     */
    getOasPropertiesDefinition : function() {
        if (PropertyEditor) {
            PropertyEditor.Util.inherit(PropertyEditor.Model.Type, ApiConnectorBuilder.configServerAndSecurity);
        }
        
        return [
            {title: get_cbuilder_msg('apiConnectorBuilder.OAS'),
                properties : [{
                    name : 'oas',
                    label : get_cbuilder_msg('apiConnectorBuilder.selectOAS'),
                    type: 'file',
                    appPath: CustomBuilder.appPath,
                    allowInput : 'false',
                    isPublic : 'false',
                    allowType : ".json;.yaml",
                    required: "true"
                },
                {
                    name : 'downloadOAS',
                    label : get_cbuilder_msg('apiConnectorBuilder.downloadOas'),
                    type: 'label',
                    control_field: 'oas',
                    control_value: '^.+$',
                    control_use_regex: 'true'
                },
                {
                    name : "config",
                    label : "",
                    type : "configServerAndSecurity",
                    script_url : CustomBuilder.contextPath+"/web/json/app"+CustomBuilder.appPath+"/plugin/org.joget.connector.ApiConnectorBuilder/service?action=oasconfig",
                    required : "True"
                }]
            }
        ];
    },
    
    /*
     * Download the OAS document and render palette
     */
    retrieveOAS : function(oas, deferreds, populatePalette) {
        var wait = $.Deferred();
        deferreds.push(wait);
        
        //if exist in cache
        if (ApiConnectorBuilder.oasCache[CustomBuilder.appPath + "/resources/" + oas] !== undefined) {
            ApiConnectorBuilder.oas = ApiConnectorBuilder.oasCache[CustomBuilder.appPath + "/resources/" + oas];
        }
        
        if (ApiConnectorBuilder.oas === undefined) {
            fetch(CustomBuilder.contextPath + "/web/app" + CustomBuilder.appPath + "/resources/" + oas)
            .then(function (response) {
                return response.text();
            })
            .then(function (data){
                ApiConnectorBuilder.updateOasDocument(data, wait, true);
        
                if (ApiConnectorBuilder.oas !== undefined) {
                    ApiConnectorBuilder.oasCache[CustomBuilder.appPath + "/resources/" + oas] = ApiConnectorBuilder.oas;
                }
            });
        } else if (populatePalette) {
            ApiConnectorBuilder.populatePalette(wait);
        } else {
            wait.resolve();
        }
    },
    
    /*
     * Update the choosen OAS document
     */
    updateOasDocument: function(content, deferred, populatePalette) {
        let oasObject = ApiConnectorBuilder.convertToObject(content);
        
        //convert swagger 2.0 to oas 3.0
        if (oasObject['swagger'] !== undefined) {
            oasObject = ApiConnectorBuilder.convertSwaggerToOas(oasObject);
        }
        
        //validate oas document
        if (oasObject['openapi'] !== undefined) {
            ApiConnectorBuilder.oas = oasObject;
            
            if (populatePalette) {
                ApiConnectorBuilder.populatePalette(deferred);
            } else {
                deferred.resolve();
            }
        } else {
            alert(get_cbuilder_msg('apiConnectorBuilder.invalidOAS'));
            ApiConnectorBuilder.showOasSelector();
            deferred.reject();
        }
    },
    
    /*
     * Based on OAS, populate all apis to palette
     */
    populatePalette: function(deferred) {
        var oas = ApiConnectorBuilder.oas;
        
        $(".components-list").find("[element-class]").each(function() {
            var className = $(this).attr("element-class");
            delete CustomBuilder.paletteElements[className];
        });
        $(".components-list").html("");
        
        if (oas.paths !== undefined) {
            for (var i in oas.paths) {
                //share paramters of all methods
                var sharedParameters = [];
                if (oas.paths[i]['parameters'] !== undefined) {
                    sharedParameters = oas.paths[i]['parameters'];
                }

                for (var j in oas.paths[i]) {
                    if (j !== "parameters") {
                        ApiConnectorBuilder.renderPath(i, j, oas.paths[i][j], sharedParameters);
                    }
                }
            }
        }
        
        deferred.resolve();
    },
    
    /*
     * Render path to palette
     */
    renderPath: function(path, method, pathData, sharedParameters) {
        //check if not deprecated
        if (pathData.deprecated) {
            return;
        }
        
        //remove special chars in ID
        var id = ApiConnectorBuilder.getPathId(path, method);
        var tag = pathData.tags.join(";");
        var label = pathData.summary;
        
        //check security 
        var security = CustomBuilder.data.properties['security_authentication'];
        var securities = [];
        for (var i in pathData.security) {
            var value = "";
            for (var j in pathData.security[i]) {
                if (value !== "") {
                    value += ";";
                }
                value += j;
            }
            securities.push(value);
        }
        if (!(security === "" || $.inArray(security, securities) !== -1)) {
            return; //security not supported
        }
        
        var methodLabel = method.toUpperCase();
        if (methodLabel === "DELETE" || methodLabel === "OPTIONS") {
            methodLabel = methodLabel.substring(0, 3);
        } 
        var icon = '<i class="'+method+'"><span style=\"font-size: 55%;top: -9px;font-weight: bold;\">'+methodLabel+'</span></i>';
        
        var defaultPropertyValues = {};
        var propertyOptions = ApiConnectorBuilder.getPathPropertyOptions(pathData, sharedParameters, defaultPropertyValues);
        
        var meta = {
            path : path,
            method : method,
            pathData : pathData,
            sharedParameters : sharedParameters,
            builderTemplate: {
                supportStyle : false,
                dragHtml : '<div class="builder-palette-element ">'+ icon + '<a>' + label + '</a></div>',
                afterAddElement : function(elementObj, component) {
                    elementObj.path = component.path;
                    elementObj.method = component.method;
                    elementObj.pathData = component.pathData;
                    elementObj.sharedParameters = component.sharedParameters;
                    
                    ApiConnectorBuilder.updateOasSchema();
                }
            }
        };
        
        CustomBuilder.initPaletteElement(tag, id, label, icon, propertyOptions, defaultPropertyValues, true, "", meta);
    },
    
    /*
     * Construct the property options to configure server & security based on the OAS document
     */
    getServerSecurityPropertyOptions: function() {
        ApiConnectorBuilder.variables = []; //reset it

        var options = [
            {
                "title": "",
                "properties": []
            }
        ];

        //server config
        if (ApiConnectorBuilder.oas.servers !== undefined) {
            options[0].properties.push({
                "label" : get_cbuilder_msg("apiConnectorBuilder.server"),
                "type" : "header"
            });
            
            if (ApiConnectorBuilder.oas.servers.length > 0) {
                var serverField = ApiConnectorBuilder.createField("server", get_cbuilder_msg('apiConnectorBuilder.selectServer'), "selectbox", null, true); 
                options[0].properties.push(serverField);

                for (var i in ApiConnectorBuilder.oas.servers) {
                    var server = ApiConnectorBuilder.oas.servers[i];

                    if (server.description !== undefined) {
                        serverField.options.push({
                            "value" : server.url,
                            "label" : server.description
                        });
                    } else {
                        serverField.options.push({
                            "value" : server.url,
                            "label" : server.url
                        });
                    }

                    //set default server to last server url
                    serverField['value'] = server.url;

                    //handle server URL variables
                    if (server.variables !== undefined) {
                        for (var vk in server.variables) {
                            var variable = server.variables[vk];

                            ApiConnectorBuilder.variables.push(vk);

                            var variableField = ApiConnectorBuilder.createField("variable_" + vk, vk, "textfield", variable.description, true, "server", server.url, false); 

                            //check if selectbox
                            if (variable.enum !== undefined) {
                                variableField['type'] = "selectbox";
                                variableField['options'] = [];

                                for (var e in variable.enum) {
                                    variableField['options'].push({
                                        "value" : variable.enum[e],
                                        "label" : variable.enum[e]
                                    });
                                }
                            }

                            //if there is default value
                            if (variable.default !== undefined) {
                                variableField['value'] = variable.default;
                            }

                            options[0].properties.push(variableField);
                        }
                    }
                }
            } else {
                var serverField = ApiConnectorBuilder.createField("server", get_cbuilder_msg('apiConnectorBuilder.server'), "textfield", null, true); 
                options[0].properties.push(serverField);
            }
        }
        
        options[0].properties.push({
            "label" : get_cbuilder_msg("apiConnectorBuilder.security"),
            "type" : "header"
        });
            
        if (ApiConnectorBuilder.oas.components !== undefined && ApiConnectorBuilder.oas.components.securitySchemes !== undefined) {
            var authField = {
                "name" : "security_authentication",
                "label" : get_cbuilder_msg('apiConnectorBuilder.selectAuthenticationMethod'),
                "type" : "selectbox",
                "options" : [
                    {
                            "value" : "",
                            "label" : ""
                    }
                ]
            };
            options[0].properties.push(authField);

            //add secuirty options 
            var checker = [];
            var populateSecurityOptions = function(securityArrs) {
                for (var i in securityArrs) {
                    var value = "";
                    var label = "";
                    for (var j in securityArrs[i]) {
                        if (ApiConnectorBuilder.oas.components.securitySchemes[j] !== undefined) {
                            if (value !== "") {
                                value += ";";
                                label += " + ";
                            }
                            value += j;
                            label += j;
                        }
                    }

                    //if not in the option yet, add it
                    if (value !== "" && $.inArray(value, checker) === -1) {
                        checker.push(value);
                        authField.options.push({
                            "value" : value,
                            "label" : label
                        });
                    }
                }
            };

            //check there is a security info
            if (ApiConnectorBuilder.oas.security !== undefined) {
                populateSecurityOptions(ApiConnectorBuilder.oas.security);
            } else {
                //loop in paths to try find security options   
                if (ApiConnectorBuilder.oas.paths !== undefined) {
                    for (var i in ApiConnectorBuilder.oas.paths) {
                        for (var j in ApiConnectorBuilder.oas.paths[i]) {
                            if (ApiConnectorBuilder.oas.paths[i][j].security !== undefined) {
                                 populateSecurityOptions(ApiConnectorBuilder.oas.paths[i][j].security);
                            }
                        }
                    }
                }
            }

            //render security fields
            for (var sc in ApiConnectorBuilder.oas.components.securitySchemes) {
                var schema = ApiConnectorBuilder.oas.components.securitySchemes[sc];
                if (schema.type === "http") {
                    if (schema.scheme === "basic") {
                        //add username & password field
                        options[0].properties.push(ApiConnectorBuilder.createField("security_schema_" + sc + "_username", get_cbuilder_msg('apiConnectorBuilder.security.username'), "textfield", null, true, "security_authentication", sc, true));
                        options[0].properties.push(ApiConnectorBuilder.createField("security_schema_" + sc + "_password", get_cbuilder_msg('apiConnectorBuilder.security.password'), "password", null, true, "security_authentication", sc, true));
                    } else if (schema.scheme === "bearer") {
                        //add token field
                        options[0].properties.push(ApiConnectorBuilder.createField("security_schema_" + sc, get_cbuilder_msg('apiConnectorBuilder.security.token'), "password", null, true, "security_authentication", sc, true));
                    }
                } else if (schema.type === "oauth2" || schema.type === "openIdConnect") {
                    var hashFlow = false;
                    var scopes = [];
                    var variables = [];

                    if (schema.type === "oauth2") {
                        var flowtype = ApiConnectorBuilder.createField("security_schema_" + sc + "_flow", get_cbuilder_msg('apiConnectorBuilder.oauth2.flow'), "selectbox", null, true, "security_authentication", sc, true);
                        options[0].properties.push(flowtype);

                        var scopeField = ApiConnectorBuilder.createField("security_schema_" + sc + "_scopes", get_cbuilder_msg('apiConnectorBuilder.oauth2.scopes'), "checkbox", null, false, "security_authentication", sc, true);


                        const regex = /\{(\w+)\}/g;
                        var findVariables = function(input) {
                            let match;
                            while ((match = regex.exec(input)) !== null) {
                                if ($.inArray(match[1], variables) === -1) {
                                    variables.push(match[1]);
                                }
                            }
                        };
                        var addScopes = function(scopesArray) {
                            for (var i in scopesArray) {
                                if ($.inArray(i, scopes) === -1) {
                                    scopes.push(i);
                                    findVariables(i);

                                    scopeField.options.push({
                                        "value" : i,
                                        "label" : scopesArray[i]
                                    });
                                }
                            }
                        };

                        if (schema.flows !== undefined) {
                            for (var flow in schema.flows) {
                                if (flow !== "implicit" && flow !== "password") { //this 2 types are lagacy and not recommended 
                                    flowtype.options.push({
                                        "value" : flow,
                                        "label" : get_cbuilder_msg('apiConnectorBuilder.oauth2.'+flow)
                                    });

                                    var flowAttr = schema.flows[flow];
                                    for (var attr in flowAttr) {
                                        if ("scopes" === attr) {
                                            addScopes(flowAttr.scopes);
                                        } else {
                                            findVariables(flowAttr[attr]);
                                        }
                                    }

                                    hashFlow = true;
                                }
                            }
                        }
                    }

                    if (hashFlow || schema.type === "openIdConnect") {
                        var controlField = "security_authentication";
                        var controlValue = sc;

                        if (schema.type === "oauth2") {
                            //Add scope
                            if (scopes.length > 0) {
                                scopeField["value"] = scopes.join(";");
                                options[0].properties.push(scopeField);

                                if (CustomBuilder.data.properties[scopeField["name"]] === undefined) {
                                    CustomBuilder.data.properties[scopeField["name"]] = scopeField["value"];
                                }
                            }

                            controlField = "security_schema_" + sc + "_flow";
                            controlValue = "authorizationCode";
                        }

                        //add client id field
                        options[0].properties.push(ApiConnectorBuilder.createField("security_schema_" + sc + "_clientId", get_cbuilder_msg('apiConnectorBuilder.security.clientId'), "textfield", null, true, "security_authentication", sc, true));

                        //add client secret field
                        options[0].properties.push(ApiConnectorBuilder.createField("security_schema_" + sc + "_clientSecret", get_cbuilder_msg('apiConnectorBuilder.security.clientSecret'), "password", null, true, "security_authentication", sc, true));

                        //add Authorization Code Challenge Method
                        var codeChallengeField = ApiConnectorBuilder.createField("security_schema_" + sc + "_codeChallenge", get_cbuilder_msg('apiConnectorBuilder.oauth2.codeChallenge'), "selectbox", null, false, controlField, controlValue, true);
                        codeChallengeField.options.push({
                            "value" : "",
                            "label" : get_cbuilder_msg('apiConnectorBuilder.oauth2.plain')
                        });
                        codeChallengeField.options.push({
                            "value" : "S256",
                            "label" : "SHA-256"
                        });
                        options[0].properties.push(codeChallengeField);
                        
                        options[0].properties.push(ApiConnectorBuilder.createField("security_schema_" + sc + "_codeVerifier", get_cbuilder_msg('apiConnectorBuilder.openId.codeVerifier'), "password", null, false, "security_schema_" + sc + "_codeChallenge", "S256", true));

                        //add callback url
                        var callbackField = ApiConnectorBuilder.createField("security_schema_" + sc + "_redirectUrl", get_cbuilder_msg('apiConnectorBuilder.oauth2.redirectUrl'), "label", null, false, controlField, controlValue, true);
                        callbackField["value"] = window.location.origin + CustomBuilder.contextPath + "/apiconnector/callback";
                        options[0].properties.push(callbackField);
                        CustomBuilder.data.properties[callbackField["name"]] = callbackField["value"];

                        if (schema.type === "openIdConnect") {
                            var openIdConnectConfigField = ApiConnectorBuilder.createField("security_schema_" + sc + "_config", get_cbuilder_msg('apiConnectorBuilder.openId.config'), "selectbox", null, false, "security_authentication", sc, true);
                            openIdConnectConfigField.options.push({
                                "value" : "",
                                "label" : get_cbuilder_msg('apiConnectorBuilder.openId.discovery')
                            });
                            openIdConnectConfigField.options.push({
                                "value" : "custom",
                                "label" : get_cbuilder_msg('apiConnectorBuilder.openId.custom')
                            });
                            options[0].properties.push(openIdConnectConfigField);

                            options[0].properties.push(ApiConnectorBuilder.createField("security_schema_" + sc + "_issuer", get_cbuilder_msg('apiConnectorBuilder.openId.issuer'), "textfield", null, false, "security_schema_" + sc + "_config", "custom", true));
                            options[0].properties.push(ApiConnectorBuilder.createField("security_schema_" + sc + "_jwksUrl", get_cbuilder_msg('apiConnectorBuilder.openId.jwksUrl'), "textfield", null, false, "security_schema_" + sc + "_config", "custom", true));
                            options[0].properties.push(ApiConnectorBuilder.createField("security_schema_" + sc + "_nonce", "", "hidden", null, true, "security_authentication", sc, true));
                            options[0].properties.push(ApiConnectorBuilder.createField("security_schema_" + sc + "_authorization_endpoint", get_cbuilder_msg('apiConnectorBuilder.openId.authorizationEndpoint'), "textfield", null, false, "security_schema_" + sc + "_config", "custom", true));
                            options[0].properties.push(ApiConnectorBuilder.createField("security_schema_" + sc + "_tokenEnpoint", get_cbuilder_msg('apiConnectorBuilder.openId.tokenEnpoint'), "textfield", null, false, "security_schema_" + sc + "_config", "custom", true));
                            options[0].properties.push(ApiConnectorBuilder.createField("security_schema_" + sc + "_scopes", get_cbuilder_msg("apiConnectorBuilder.oauth2.scopes"), "textfield", null, false, "security_schema_" + sc + "_config", "custom", true));
                            options[0].properties.push(ApiConnectorBuilder.createField("security_schema_" + sc + "_responseType", get_cbuilder_msg("apiConnectorBuilder.openId.responseType"), "textfield", null, false, "security_schema_" + sc + "_config", "custom", true));
                        }

                        //add Authorization Code
                        options[0].properties.push(ApiConnectorBuilder.createField("security_schema_" + sc + "_authorizationCode", get_cbuilder_msg('apiConnectorBuilder.oauth2.authorizationCode'), "password", null, true, controlField, controlValue, true));

                        //add Client Authentication field
                        var clientAuthenticationField = ApiConnectorBuilder.createField("security_schema_" + sc + "_clientAuthentication", get_cbuilder_msg('apiConnectorBuilder.oauth2.clientAuthentication'), "selectbox", null, false, "security_authentication", sc, true);
                        clientAuthenticationField.options.push({
                            "value" : "",
                            "label" : get_cbuilder_msg('apiConnectorBuilder.oauth2.sendAsBasicAuthHeader')
                        });
                        clientAuthenticationField.options.push({
                            "value" : "sendInBody",
                            "label" : get_cbuilder_msg('apiConnectorBuilder.oauth2.sendInBody')
                        });
                        options[0].properties.push(clientAuthenticationField);

                        //add variables
                        for (var i in variables) {
                            if (CustomBuilder.data.properties["variable_" + variables[i]] === undefined) { //not exist in server setting
                                options[0].properties.push(ApiConnectorBuilder.createField("variable_" + variables[i], variables[i], "textfield", null, true, "security_authentication", sc, true));
                            }
                        }
                    }
                } else if (schema.type === "apiKey"){
                    options[0].properties.push(ApiConnectorBuilder.createField("security_schema_" + sc, sc, "password", schema.description, true, "security_authentication", sc, true));
                }
            }
        }
        
        options[0].properties.push({
            "name" : "headers",
            "label" : get_cbuilder_msg('apiConnectorBuilder.headers'),
            "type" : "grid",
            "columns": [
                {
                    "key" : "name", 
                    "label" : get_cbuilder_msg('cbuilder.name')
                },
                {
                    "key" : "value", 
                    "label" : get_cbuilder_msg('apiConnectorBuilder.value')
                }
            ]
        });

        return options;
    },
    
    /*
     * Construct the property options to configure path based on the OAS document
     */
    getPathPropertyOptions: function(data, sharedParameters, pathProperties) {
        var options = [
            {
                "title": data.summary,
                "properties": [
                    {
                        "name" : "name",
                        "label" : get_cbuilder_msg('cbuilder.name'),
                        "type" : "textfield"
                    },
                    {
                        "name" : "description",
                        "label" : get_cbuilder_msg('ubuilder.description'),
                        "type" : "codeeditor",
                        "mode" : "text"
                    }
                ]
            }
        ];


        if (data.description !== undefined && data.description !== "" && pathProperties['description'] === undefined) {
            pathProperties['description'] = data.description;
        }

        if (data.requestBody !== undefined) {
            //find request body schema or example
            if (data.requestBody.content !== undefined) {
                var content = null;
                var contentType = 'application/json';

                if (data.requestBody.content['application/json'] !== undefined) {
                    content = data.requestBody.content['application/json']; //use json if exist
                } else {
                    //find first media type
                    for (var i in data.requestBody.content) {
                        content = data.requestBody.content[i];
                        contentType = i;
                        break;
                    }
                }

                var requestBodyInputsField = {
                    "name" : "requestBodyInputs",
                    "label" : get_cbuilder_msg('apiConnectorBuilder.requestBody.fields'),
                    "type" : "gridfixedrow",
                    "columns": [
                        {
                            "key" : "name", 
                            "label" : get_cbuilder_msg('cbuilder.name')
                        },
                        {
                            "key" : "customLabel", 
                            "label" : get_cbuilder_msg('apiConnectorBuilder.customLabel')
                        },
                        {
                            "key" : "defaultValue", 
                            "label" : get_cbuilder_msg('apiConnectorBuilder.defaultValue')
                        },
                        {
                            "key" : "mode", 
                            "label" : get_cbuilder_msg('apiConnectorBuilder.mode'),
                            "options" : [
                                {
                                    "value" : "",
                                    "label" : get_cbuilder_msg('apiConnectorBuilder.mode.default')
                                },
                                {
                                    "value" : "mandatory",
                                    "label" : get_cbuilder_msg('apiConnectorBuilder.mode.mandatory')
                                },
                                {
                                    "value" : "hide",
                                    "label" : get_cbuilder_msg('apiConnectorBuilder.mode.hide')
                                }
                            ]
                        }
                    ],
                    "rows": [
                    ]
                };

                var example;
                if (content.schema !== undefined) {
                    var schema;
                    if (content.schema['$ref'] !== undefined) {
                        schema = ApiConnectorBuilder.getSchemaByRef(null, content.schema['$ref']);
                    } else {
                        schema = content.schema;
                    }
                    example = ApiConnectorBuilder.getExampleBySchema(schema);
                }

                var exampleJson = JSON.stringify(example, null, 4);

                if (contentType.indexOf("json") !== -1) {
                    var requestBodyTypeField = {
                        "name" : "requestBodyType",
                        "label" : get_cbuilder_msg('apiConnectorBuilder.requestBodyInputType'),
                        "type" : "selectbox",
                        "options" : [
                            {
                                "value" : "payload",
                                "label" : get_cbuilder_msg('apiConnectorBuilder.requestBodyInputType.payload')
                            },
                            {
                                "value" : "fields",
                                "label" : get_cbuilder_msg('apiConnectorBuilder.requestBodyInputType.fields')
                            }
                        ],
                        "value" : "payload",
                        "required" : "true"
                    };
                    options[0].properties.push(requestBodyTypeField);

                    var requestBodyField = {
                        "name" : "requestBody",
                        "label" : get_cbuilder_msg('apiConnectorBuilder.requestBody.format'),
                        "type" : "codeeditor",
                        "mode" : "json",
                        "control_field" : 'requestBodyType',
                        "control_value" : "payload",
                        "control_use_regex" : 'true'
                    };
                    options[0].properties.push(requestBodyField);

                    //set default format
                    requestBodyField['value'] = exampleJson;
                    if (pathProperties['requestBody'] === undefined) {
                        pathProperties['requestBody'] = exampleJson;
                    }

                    //add dependency control field 
                    requestBodyInputsField["control_field"] = "requestBodyType";
                    requestBodyInputsField["control_value"] = "fields";
                    requestBodyInputsField["control_use_regex"] = "true";
                } else {
                    //if not json playload
                    if (pathProperties['requestBodyType'] === undefined) {
                        pathProperties['requestBodyType'] = "fields";
                    }
                }

                options[0].properties.push(requestBodyInputsField);

                //populate field by example
                var populateFieldsByExample = function(data, prefix) {
                    if (Array.isArray(data)) {
                        if (typeof data[0] === 'object') {
                            populateFieldsByExample(data[0], prefix);
                        }
                    } else {
                        //loop all properties
                        for (var i in data) {
                            var key = ((prefix !== undefined)?prefix+"_":"") + i;
                            if (typeof data[i] === 'object') {
                                requestBodyInputsField.rows.push({
                                    "label" : key
                                });
                                populateFieldsByExample(data[i], key);
                            } else {
                                requestBodyInputsField.rows.push({
                                    "label" : key
                                });
                            }
                        }
                    }
                };
                populateFieldsByExample(example);
            }
        }

        var parameters = [];
        if (data.parameters !== undefined) {
            parameters = $.extend(parameters, data.parameters);
        }
        if (sharedParameters.length > 0) { //append extra shared parameters
            parameters = $.extend(parameters, sharedParameters);
        }

        if (parameters.length > 0) {
            var parametersField = {
                "name" : "parameters",
                "label" : get_cbuilder_msg('apiConnectorBuilder.parameters'),
                "type": "gridfixedrow",
                "columns": [
                    {
                        "key" : "name", 
                        "label" : get_cbuilder_msg('cbuilder.name')
                    },
                    {
                        "key" : "customLabel", 
                        "label" : get_cbuilder_msg('apiConnectorBuilder.customLabel')
                    },
                    {
                        "key" : "defaultValue", 
                        "label" : get_cbuilder_msg('apiConnectorBuilder.defaultValue')
                    },
                    {
                        "key" : "mode", 
                        "label" : get_cbuilder_msg('apiConnectorBuilder.mode'),
                        "options" : [
                            {
                                "value" : "",
                                "label" : get_cbuilder_msg('apiConnectorBuilder.mode.default')
                            },
                            {
                                "value" : "mandatory",
                                "label" : get_cbuilder_msg('apiConnectorBuilder.mode.mandatory')
                            },
                            {
                                "value" : "hide",
                                "label" : get_cbuilder_msg('apiConnectorBuilder.mode.hide')
                            }
                        ]
                    }
                ],
                "rows": [
                ]
            };
            options[0].properties.push(parametersField);

            var paramValues = [];
            for (var p in parameters) {
                var parameter = parameters[p];
                if (parameter['$ref'] !== undefined) {
                    parameter = ApiConnectorBuilder.getSchemaByRef(null, parameter['$ref']);
                }

                parametersField.rows.push({
                    "label" : parameter.name
                });
                paramValues.push({name:parameter.name});
            }

            //preset value if not exist
            if (pathProperties.parameters === undefined) {
                pathProperties.parameters = paramValues;
            }
        }

        return options;
    },
    
    /*
     * populate the code back to the field after completed the Oauth2 Authorization Code flow
     */
    readingAuthorizationCode: function(button) {
        var loop = function() {
            var compareState = $(button).data("state");
            var value = localStorage.getItem(compareState);
            if (value) {
                var data = JSON.parse(value);
                if (data['error'] !== undefined && data['error'] !== "") {
                    alert(data['error']);
                } else {
                    $(button).prev("input").val(data['code']).trigger("change");
                }
                $.unblockUI();
                
                localStorage.removeItem(compareState);
                return;
            }
            setTimeout(function(){
                loop();
            }, 5);
        };
        loop();
    },
    
    /*
     * Convert the uploaded json or yaml file string to object
     */
    convertToObject: function(content) {
        //check is it json format
        if (content.trim().indexOf("{") === 0) {
            //convert json to object
            return JSON.parse(content);
        } else {
            //convert yaml to object
            return jsyaml.load(content);
        }
    },
    
    /*
     * Convert Swagger 2.0 to OAS 3.0
     */
    convertSwaggerToOas: function(oas2) {
        // Helper function to convert operations
        const convertParameter = function(parameter) {
            var convertedParameter = {};
            if (parameter["$ref"] !== undefined) { //handle parameter with ref to schema
                return parameter;
            } else {
                var props = ["name", "in", "required", "schema"];
                for (var i in props) {
                    if (parameter[props[i]] !== undefined) {
                        convertedParameter[props[i]] = parameter[props[i]];
                        delete parameter[props[i]];
                    }
                }
                if (convertedParameter["schema"] === undefined) {
                    delete parameter["collectionFormat"];
                    convertedParameter["schema"] = parameter;
                }
            }
            return convertedParameter;
        };

        // Helper function to convert form data to schema
        const convertFormData = function(formData) {
            var required = [];
            var convertedSchema = {
                type: "object",
                properties : {}
            };
            for (var i in formData) {
                var name = formData[i].name;
                delete formData[i].name;
                delete formData[i].in;
                if (formData[i].required) {
                    required.push(name);
                }
                if (formData[i].type === "file") {
                    formData[i].type = "string";
                    formData[i].format = "binary";
                }
                delete formData[i].required;
                convertedSchema.properties[name] = formData[i];
            }
            if (required.length > 0) {
                convertedSchema.required = required;
            }
            return convertedSchema;
        };

        // Helper function to convert operations
        const convertOperation = function(operation) {
            var convertedOperation = {
                responses: {}
            };
            var props = ["summary", "description", "operationId", "security", "tags"];
            for (var i in props) {
                if (operation[props[i]] !== undefined) {
                    convertedOperation[props[i]] = operation[props[i]];
                }
            }
            var parameters = [];
            var bodySchema = null;
            var formData = [];
            for (var i in operation.parameters) {
                var p = operation.parameters[i];
                if (p.name === "body") {
                    if (p.description === undefined) {
                        convertedOperation.requestBody.description = p.description;
                    }
                    if (p.required === undefined) {
                        convertedOperation.requestBody.required = p.required;
                    }
                    if (operation.consumes === undefined) {
                        operation.consumes = ["application/json"];
                    }
                } else if (p.in === "formData") {
                    formData.push(p);
                } else {
                    parameters.push(p);
                }
            }
            if (formData.length > 0) {
                bodySchema = convertFormData(formData);
                if (operation.consumes === undefined) {
                    operation.consumes = ["application/x-www-form-urlencoded"];
                }
            }
            if (bodySchema !== null && operation.consumes !== undefined) {
                convertedOperation.requestBody = {
                    content : {}
                };
                operation.consumes.forEach(mediaType => {
                    convertedOperation.requestBody.content[mediaType] = {
                        schema : bodySchema
                    };
                });
            }
            if (parameters.length > 0) {
                convertedOperation.parameters = [];
                for (var i in parameters) {
                    convertedOperation.parameters[i] = convertParameter(parameters[i]);
                }
            }

            if (operation.responses) {
                for (let statusCode in operation.responses) {
                    const response = operation.responses[statusCode];
                    convertedOperation.responses[statusCode] = {
                        description: response.description,
                        content : {}
                    };

                    if (response.schema !== undefined && operation.produces !== undefined) {
                        operation.produces.forEach(mediaType => {
                            convertedOperation.responses[statusCode].content[mediaType] = {
                                schema : response.schema
                            };
                        });
                    }
                }
            }
            return convertedOperation;
        };

        // Create an empty OAS 3.0 object
        let oas3 = {
            openapi: "3.0.0",
            info: oas2.info,
            servers: [],
            paths: {},
            components: {}
        };

        try {

            // Convert servers
            if (oas2.host) {
                oas3.servers.push({
                    url: `${oas2.schemes && oas2.schemes[0] ? oas2.schemes[0] : 'https'}://${oas2.host}${oas2.basePath || '/'}`
                });
            }
            if (oas2["x-servers"]) {
                for (let s in oas2["x-servers"]) {
                    if (oas2["x-servers"][s]["url"]) {
                        oas3.servers.push({
                            url: oas2["x-servers"][s]["url"]
                        });
                    }
                }
            }

            // Convert paths
            for (let path in oas2.paths) {
                const pathObject = oas2.paths[path];
                oas3.paths[path] = {};
                for (let method in pathObject) {
                    const operation = pathObject[method];
                    oas3.paths[path][method] = convertOperation(operation);
                }
            }

            // Convert components
            if (oas2.definitions !== undefined) {
                oas3.components.schemas = oas2.definitions;
            }

            // Convert Security
            if (oas2.securityDefinitions !== undefined) {
                oas3.components.securitySchemes = oas2.securityDefinitions;
                for (var i in oas3.components.securitySchemes) {
                    if ("oauth2" === oas3.components.securitySchemes[i].type) {
                        var flow = oas3.components.securitySchemes[i].flow;
                        delete oas3.components.securitySchemes[i].flow;
                        delete oas3.components.securitySchemes[i].type;
                        var temp = oas3.components.securitySchemes[i];
                        oas3.components.securitySchemes[i] = {
                            type : "oauth2",
                            flows: {}
                        };
                        oas3.components.securitySchemes[i].flows[flow] = temp;
                    }
                }
            }

            // Convert shared parameters
            if (oas2.parameters !== undefined) {
                oas3.components.parameters = oas2.parameters;

                //convert parameters
                for (var i in oas3.components.parameters) {
                    oas3.components.parameters[i] = convertParameter(oas3.components.parameters[i]);
                }
            }

            // Convert responses
            if (oas2.responses !== undefined) {
                oas3.components.responses = oas2.responses;
            }

            // Convert tags
            if (oas2.tags !== undefined) {
                oas3.tags = oas2.tags;
            }

            //update all references 
            var oas3String = JSON.stringify(oas3);
            oas3String = oas3String.replaceAll("/definitions/", "/components/schemas/");
            oas3String = oas3String.replaceAll("/parameters/", "/components/parameters/");
            oas3String = oas3String.replaceAll("/responses/", "/components/responses/");

            return JSON.parse(oas3String);
        } catch (err) {
            console.error(err);
        }

        return oas2;
    },

    /*
     * Get schema by ref path
     */
    getSchemaByRef: function(data, ref) {
        if (data === null || data === undefined) {
            data = ApiConnectorBuilder.oas;
        }

        if (ref.indexOf("#/") === 0) {
            ref = ref.substring(2);
        }

        var index = ref.indexOf("/");
        if (index !== -1) {
            var name = ref.substring(0, index);
            ref = ref.substring(index + 1);
            return ApiConnectorBuilder.getSchemaByRef(data[name], ref);
        } else {
            return data[ref];
        }
    },

    /*
     * Build a sample object by schema
     */
    getExampleBySchema: function(schema, key) {
        var example;
        if (schema.type === "array") {
            example = [];
            if (schema.items !== undefined) {
                var arrItemExample;
                if (schema.items['$ref'] !== undefined) {
                    var itemSchema = ApiConnectorBuilder.getSchemaByRef(null, schema.items['$ref']);
                    arrItemExample = ApiConnectorBuilder.getExampleBySchema(itemSchema);
                } else {
                    arrItemExample = ApiConnectorBuilder.getExampleBySchema(schema.items);
                }
                example.push(arrItemExample);
            }
        } else if (schema.type === "object") {
            example = {};

            if (schema.properties !== undefined) {
                for (var i in schema.properties) {
                    example[i] = ApiConnectorBuilder.getExampleBySchema(schema.properties[i], i);
                }
            }
        } else if (schema.type === "string")  {
            if (schema.description !== undefined) {
                return schema.description;
            } else if (key !== undefined) {
                return key;
            } else {
                return "string";
            }
        } else if (schema.type === "number")  {
            return 1.0;
        } else if (schema.type === "integer")  {
            return 1;
        } else if (schema.type === "boolean")  {
            return true;
        }

        return example;
    },

    createField : function(name, label, type, description, required, controlField, controlValue, controlRegex) {
        var field = {
            "name" : name,
            "label" : label,
            "type" : type
        };

        if (type === "selectbox" || type === "checkbox" || type === "radio") {
            field["options"] = [];
        }

        if (required) {
            field["required"] = "true";
        }
        if (description !== undefined && description !== null) {
            field["description"] = description;
        }
        if (controlField !== undefined && controlField !== null) {
            field["control_field"] = controlField;
            field["control_value"] = controlValue;
            field["control_use_regex"] = (controlRegex)?"true":"false";
        }

        return field;
    },

    /*
     * generate id for path
     */
    getPathId : function(path, method) {
        var id = method + "_" + path.replace(/\//g, '_slash_');
        id = id.replace(/\{/g, '_ob_');
        id = id.replace(/\}/g, '_cb_');
        return id;
    },

    /*
     * fill the variable found in the input string with proprties values
     */
    fillVariables: function(input, props, prefix) {
        if (input.indexOf("{") !== -1 && input.indexOf("}") !== -1) { //having variables
            // Regex pattern to match variables enclosed in curly braces
            var regex = /\{([^}]+)\}/g;

            // Use match() method to find all variables
            var matches = input.match(regex);

            // If matches found, iterate over each match
            if (matches) {
                matches.forEach(function(match) {
                    var key = match.substring(1, match.length -1);
                    if (props[prefix+key] !== undefined) {
                        input = input.replace(match, props[prefix+key]);
                    }
                });
            }
        }
        return input;
    },

    /*
     * Generate a random string for Oauth2 authentication
     */
    generateRandomString: function(length) {
        var text = '';
        var possible = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-._~';

        for (var i = 0; i < length; i++) {
            text += possible.charAt(Math.floor(Math.random() * possible.length));
        }

        return text;
    },

    /*
     * Generate a Code Challenge from verifier using base64url(sha256(code_verifier))
     */
    generateCodeChallengeFromVerifier: async function(codeVerifier) {
        const encoder = new TextEncoder();
        const data = encoder.encode(codeVerifier);
        const hashed = await window.crypto.subtle.digest('SHA-256', data);
        return btoa(String.fromCharCode.apply(null, new Uint8Array(hashed)))
            .replace(/\+/g, '-')
            .replace(/\//g, '_')
            .replace(/=/g, '');
    },
    
    /**
     * Remove unused schema based on selected APIs
     */
    updateOasSchema: function() {
        var data = CustomBuilder.data;
        const oasString = JSON.stringify(data.elements);

        var found = [];
        var copyComponents = $.extend(true, {}, ApiConnectorBuilder.oas.components);
        
        var check = function(checkingStr) {
            var intCount = found.length;
            var temp = {
                schemas : {},
                parameters : {},
                responses : {}
            };
            
            for (var i in copyComponents.schemas) {
                if (checkingStr.indexOf("/components/schemas/"+i) !== -1 && found.indexOf("/components/schemas/"+i) === -1) {
                    temp.schemas[i] = $.extend(true, {}, copyComponents.schemas[i]);
                    found.push("/components/schemas/"+i);
                }
            }

            for (var i in copyComponents.parameters) {
                if (checkingStr.indexOf("/components/parameters/"+i) !== -1 && found.indexOf("/components/parameters/"+i) === -1) {
                    temp.parameters[i] = $.extend(true, {}, copyComponents.parameters[i]);
                    found.push("/components/parameters/"+i);
                }
            }

            for (var i in copyComponents.responses) {
                if (checkingStr.indexOf("/components/responses/"+i) !== -1 && found.indexOf("/components/responses/"+i) === -1) {
                    temp.responses[i] = $.extend(true, {}, copyComponents.schemas[i]);
                    found.push("/components/responses/"+i);
                }
            }
            
            if (found.length > intCount) {
                var newString = JSON.stringify(temp);
                if (newString.indexOf("/components/") !== -1) {
                    check(newString);
                }
            }
        };
        
        check(oasString);
        
        var copyComponents = $.extend(true, {}, ApiConnectorBuilder.oas.components);
        
        for (var i in copyComponents.schemas) {
            if (found.indexOf("/components/schemas/"+i) === -1) {
                delete copyComponents.schemas[i];
            }
        }

        for (var i in copyComponents.parameters) {
            if (found.indexOf("/components/parameters/"+i) === -1) {
                delete copyComponents.parameters[i];
            }
        }

        for (var i in copyComponents.responses) {
            if (found.indexOf("/components/responses/"+i) === -1) {
                delete copyComponents.schemas[i];
            }
        }
        
        CustomBuilder.data.components = copyComponents;
    },
    
    /*
     * remove dynamically added items    
     */            
    unloadBuilder : function() {
        $("#oas-btn").remove();
    },
    
    /*
     * A field used by property editor to configure server & security
     */
    configServerAndSecurity : {
        shortname : "configServerAndSecurity",
        isInit: true,
        getData: function(useDefault) {
            var thisObj = this;
            
            var properties = new Object();
            
            var fields = thisObj.fields;
            if (fields !== undefined) {
                $.each(fields, function(i, property) {
                    var type = property.propertyEditorObject;

                    if (!type.isHidden()) {
                        var data = type.getData(false);

                        //handle Hash Field
                        if (data !== null && data['HASH_FIELD'] !== null && data['HASH_FIELD'] !== undefined) {
                            if (properties['PROPERTIES_EDITOR_METAS_HASH_FIELD'] === undefined) {
                                properties['PROPERTIES_EDITOR_METAS_HASH_FIELD'] = data['HASH_FIELD'];
                            } else {
                                properties['PROPERTIES_EDITOR_METAS_HASH_FIELD'] += ";" + data['HASH_FIELD'];
                            }
                            delete data['HASH_FIELD'];
                        }

                        if (data !== null) {
                            properties = $.extend(properties, data);
                        }
                    }
                });
            }
            
            if (!thisObj.isInit && properties['server'] === undefined) {
                properties['server'] = '_';
            }
            
            thisObj.isInit = false;
            return properties;
        },
        validate: function(data, errors, checkEncryption) {
            var thisObj = this;
            var deferreds = [];
            var checkEncryption = false;
            
            //remove previous error message
            $("#" + this.id + " .property-input-error").remove();
            $("#" + this.id + " .property-editor-page-errors").remove();

            if (!this.isHidden()) {
                var fields = thisObj.fields;

                if (fields !== undefined && fields !== null) {
                    $.each(fields, function(i, property) {
                        var type = property.propertyEditorObject;
                        if (!type.isHidden()) {
                            var deffers = type.validate(data, errors, false);
                            if (deffers !== null && deffers !== undefined && deffers.length > 0) {
                                deferreds = $.merge(deferreds, deffers);
                            }
                        }
                    });
                }
            }

            return deferreds;
        },
        renderField: function() {
            return '';
        },
        initScripting: function() {
            var field = this;
            
            var propertiesContainer = $('<div class="serverSecurityConfigContainer"></div>');
            
            $("#" + this.id + "_input").closest('.property-editor-property').hide().after(propertiesContainer);
            
            var oasDocField = $(field.editor).find('[property-name="oas"] .property-input input');
            
            $(oasDocField).off("change.ssconfig").on("change.ssconfig", function(){
                field.renderFields(oasDocField.val());
            });
            field.renderFields(oasDocField.val());
        },
        renderFields: function(oas) {
            var thisObj = this;
            
            if (oas === undefined || oas === "") {
                return;
            }
            
            delete ApiConnectorBuilder.oas;
            
            var deferreds = [];
        
            ApiConnectorBuilder.retrieveOAS(oas, deferreds, false);  
            
            var data = CustomBuilder.data.properties;

            $.when.apply($, deferreds).then(function() {
                
                var options = ApiConnectorBuilder.getServerSecurityPropertyOptions();
                var fields = options[0].properties;
                thisObj.fields = fields;
                var fieldsHolder = {};
                
                //render fields
                var html = "";
                if (fields !== null && fields !== undefined) {
                    $.each(fields, function(i, property) {
                        html += thisObj.renderProperty(i + 3, property, data, fieldsHolder);
                    });
                }
                $(thisObj.editor).find(".serverSecurityConfigContainer").html(html);
                
                //init fields scripting
                if (fields !== null && fields !== undefined) {
                    $.each(fields, function(i, property) {
                        var type = property.propertyEditorObject;
                        type.initScripting();
                        type.initDefaultScripting();
                    });
                }

                //init fields tooltip
                $(thisObj.editor).find(".serverSecurityConfigContainer .property-label-description").each(function(){
                    if (!$(this).hasClass("tooltipstered")) {
                        $(this).tooltipster({
                            contentCloning: false,
                            side : 'right',
                            interactive : true
                        });
                    }
                });
                
                //bind dynamic field events
                var triggerChangeFields = [];
                var page = {
                    fields : fieldsHolder,
                    editorObject : thisObj.editorObject,
                    parentId : ""
                };
                $(thisObj.editor).find(".serverSecurityConfigContainer [data-control_field][data-control_value]").each(function() {
                    PropertyEditor.Util.bindDynamicOptionsEvent($(this), page, triggerChangeFields);
                });
                $(thisObj.editor).find(".serverSecurityConfigContainer [data-required_control_field][data-required_control_value]").each(function() {
                    PropertyEditor.Util.bindDynamicRequiredEvent($(this), page, triggerChangeFields);
                });

                for (var i in triggerChangeFields) {
                    $(thisObj.editor).find("[name=\"" + triggerChangeFields[i] + "\"]").trigger("change");
                }
                
                var propertiesContainer = $(thisObj.editor).find(".serverSecurityConfigContainer");
                
                setTimeout(function(){
                    //add button when there is authorizationCode field
                    $(propertiesContainer).find(".getAuthorizationCode").remove();
                    $(propertiesContainer).find("[name$='authorizationCode']").after('<button class="getAuthorizationCode btn btn-secondary btn-sm" style="margin-top:5px;">'+get_cbuilder_msg('apiConnectorBuilder.oauth2.getAuthorizationCode')+'</button>');
                },10);


                //Handle OAuth2 Authorization Code flow
                $(propertiesContainer).off("click.authorizationCode", "button.getAuthorizationCode")
                    .on("click.authorizationCode", "button.getAuthorizationCode", function(){
                        var button = $(this);
                        var time = (new Date()).getTime() + "";
                        $.blockUI({ css: { 
                            border: 'none', 
                            padding: '15px', 
                            backgroundColor: '#000', 
                            '-webkit-border-radius': '10px', 
                            '-moz-border-radius': '10px', 
                            opacity: .3, 
                            color: '#fff' 
                        }, message : "<i class='icon-spinner icon-spin icon-2x fas fa-spinner fa-spin fa-2x'></i>" }); 
                        
                        $(button).data("state", time);

                        var propData = thisObj.editorObject.getData();

                        var auth = $(this).prev("input").attr('id');
                        auth = auth.substring(auth.indexOf("security_schema_") + 16, auth.length - 18);
                        var securitySchemas = ApiConnectorBuilder.oas.components.securitySchemes[auth];

                        var param = {
                            "response_type" : ["code"],
                            "client_id" : [propData['security_schema_'+auth+'_clientId']],
                            "redirect_uri" : [propData['security_schema_'+auth+'_redirectUrl']],
                            "state" : [time]
                        };

                        var deferreds = [];
                        var wait = $.Deferred();
                        deferreds.push(wait);

                        var url = null;
                        if (securitySchemas.type === "oauth2") {
                            var flow = securitySchemas.flows['authorizationCode'];
                            url = flow['authorizationUrl'];
                            if (url.indexOf("/") === 0) {
                                var serverUrl = propData['server'];
                                if (serverUrl.substring(serverUrl.length -1) === "/") {
                                    serverUrl = serverUrl.substring(0, serverUrl.length -1);
                                }
                                url = serverUrl + url;
                            }
                            param["scope"] = [ApiConnectorBuilder.fillVariables(propData['security_schema_'+auth+'_scopes'].replace(";", " "), propData, "variable_")];

                            wait.resolve();
                        } else if (securitySchemas.type === "openIdConnect") {
                            if (propData['security_schema_'+auth+'_authorization_endpoint'] !== undefined && propData['security_schema_'+auth+'_authorization_endpoint'] !== "") {
                                url = propData['security_schema_'+auth+'_authorization_endpoint'];
                                param["scope"] = [propData['security_schema_'+auth+'_scopes']];
                                param["response_type"] = [propData['security_schema_'+auth+'_responseType']];
                                wait.resolve();
                            } else {
                                var discoveryUrl = securitySchemas.openIdConnectUrl;
                                if (discoveryUrl.indexOf("/") === 0) {
                                    var serverUrl = propData['server'];
                                    if (serverUrl.substring(serverUrl.length -1) === "/") {
                                        serverUrl = serverUrl.substring(0, serverUrl.length -1);
                                    }
                                    discoveryUrl = serverUrl + discoveryUrl;
                                    discoveryUrl = ApiConnectorBuilder.fillVariables(discoveryUrl, propData, "variable_");
                                }

                                var internalUrl = window.location.origin + CustomBuilder.contextPath + "/apiconnector/openIdDiscovery?url=" + encodeURIComponent(discoveryUrl);
                                fetch(internalUrl)
                                    .then(response => {
                                        if (!response.ok) {
                                            throw new Error('Network response was not ok');
                                        }
                                        return response.json();
                                    })
                                    .then(data => {
                                        if (data.issuer !== undefined) {
                                            param["scope"] = [data.scopes_supported.join(" ")];
                                            param["response_type"] = [data.response_types_supported[0]];
                                            param["nonce"] = [ApiConnectorBuilder.generateRandomString(15)];

                                            $(propertiesContainer).find("[name$='_security_schema_"+auth+"_issuer']").val(data.issuer);
                                            $(propertiesContainer).find("[name$='_security_schema_"+auth+"_jwksUrl']").val(data.jwks_uri);
                                            $(propertiesContainer).find("[name$='_security_schema_"+auth+"_authorization_endpoint']").val(data.authorization_endpoint);
                                            $(propertiesContainer).find("[name$='_security_schema_"+auth+"_tokenEnpoint']").val(data.token_endpoint);
                                            $(propertiesContainer).find("[name$='_security_schema_"+auth+"_nonce']").val(param["nonce"][0]);
                                            $(propertiesContainer).find("[name$='_security_schema_"+auth+"_scopes']").val(param["scope"][0]);
                                            $(propertiesContainer).find("[name$='_security_schema_"+auth+"_responseType']").val(param["response_type"][0]).trigger("change");

                                            url = data.authorization_endpoint;
                                            wait.resolve();
                                        } else {
                                            alert(get_cbuilder_msg("apiConnectorBuilder.openId.configError"));
                                            $(propertiesContainer).find("[name$='_security_schema_"+auth+"_config']").val("custom").trigger("change").trigger("chosen:updated");
                                        }
                                    })
                                    .catch(error => {
                                        alert(get_cbuilder_msg("apiConnectorBuilder.openId.configError"));
                                        $(propertiesContainer).find("[name$='_security_schema_"+auth+"_config']").val("custom").trigger("change").trigger("chosen:updated");
                                    });
                            }
                        }

                        $.when.apply($, deferreds).then(async function() {
                            if (url !== null) {
                                if (propData['security_schema_'+auth+'_codeChallenge'] === "S256") {
                                    var codeVerifier = ApiConnectorBuilder.generateRandomString(128);
                                    var codeChallenge = await ApiConnectorBuilder.generateCodeChallengeFromVerifier(codeVerifier);
                                    $(propertiesContainer).find("[name$='_security_schema_"+auth+"_codeVerifier']").val(codeVerifier);
                                    param['code_challenge'] = [codeChallenge];
                                    param['code_challenge_method'] = ["S256"];
                                }
                                var authUrl = url + "?" + UrlUtil.constructUrlQueryString(param);
                                authUrl = ApiConnectorBuilder.fillVariables(authUrl, propData, "variable_");

                                var authWindow = window.open(authUrl);
                                
                                ApiConnectorBuilder.readingAuthorizationCode($(button));
                            }
                        });
                    });
                
            });    
        },
        renderProperty: function(i, property, values, fieldsHolder) {
            var type = property.propertyEditorObject;

            if (type === undefined) {
                var value = null;
                if (values !== null && values !== undefined && values[property.name] !== undefined) {
                    value = values[property.name];
                } else if (property.value !== undefined && property.value !== null) {
                    value = property.value;
                }

                type = PropertyEditor.Util.getTypeObject(this, i, "", property, value, null);
                property.propertyEditorObject = type;
                
                fieldsHolder[property.name] = type;
            }

            if (type !== null) {
                return type.render();
            }
            return "";
        }
    }
};
