/*
 * Decompiled with CFR 0.152.
 */
package org.glowroot.agent.embedded.shaded.org.glowroot.ui;

import java.math.BigInteger;
import java.security.GeneralSecurityException;
import java.security.SecureRandom;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.glowroot.agent.embedded.shaded.org.glowroot.common2.config.LdapConfig;
import org.glowroot.agent.embedded.shaded.org.glowroot.common2.config.RoleConfig;
import org.glowroot.agent.embedded.shaded.org.glowroot.common2.config.UserConfig;
import org.glowroot.agent.embedded.shaded.org.glowroot.common2.repo.ConfigRepository;
import org.glowroot.agent.embedded.shaded.org.glowroot.common2.repo.PasswordHash;
import org.glowroot.agent.embedded.shaded.org.glowroot.ui.CommonHandler;
import org.glowroot.agent.embedded.shaded.org.glowroot.ui.ImmutableAuthentication;
import org.glowroot.agent.embedded.shaded.org.glowroot.ui.ImmutableSession;
import org.glowroot.agent.embedded.shaded.org.glowroot.ui.LayoutService;
import org.glowroot.agent.embedded.shaded.org.glowroot.ui.LdapAuthentication;
import org.glowroot.agent.embedded.shaded.org.glowroot.ui.SessionMapFactory;
import org.glowroot.agent.shaded.com.google.common.base.Preconditions;
import org.glowroot.agent.shaded.com.google.common.base.Splitter;
import org.glowroot.agent.shaded.com.google.common.collect.ImmutableSet;
import org.glowroot.agent.shaded.com.google.common.collect.Sets;
import org.glowroot.agent.shaded.com.google.common.net.MediaType;
import org.glowroot.agent.shaded.io.netty.handler.codec.http.HttpHeaderNames;
import org.glowroot.agent.shaded.io.netty.handler.codec.http.HttpResponseStatus;
import org.glowroot.agent.shaded.io.netty.handler.codec.http.cookie.Cookie;
import org.glowroot.agent.shaded.io.netty.handler.codec.http.cookie.DefaultCookie;
import org.glowroot.agent.shaded.io.netty.handler.codec.http.cookie.ServerCookieDecoder;
import org.glowroot.agent.shaded.io.netty.handler.codec.http.cookie.ServerCookieEncoder;
import org.glowroot.agent.shaded.org.glowroot.common.util.Clock;
import org.glowroot.agent.shaded.org.slf4j.Logger;
import org.glowroot.agent.shaded.org.slf4j.LoggerFactory;
import org.immutables.serial.Serial;
import org.immutables.value.Value;

class HttpSessionManager {
    private static final Logger logger = LoggerFactory.getLogger(HttpSessionManager.class);
    private static final Logger auditLogger = LoggerFactory.getLogger((String)"audit");
    private final boolean central;
    private final boolean offlineViewer;
    private final ConfigRepository configRepository;
    private final Clock clock;
    private final LayoutService layoutService;
    private final SecureRandom secureRandom = new SecureRandom();
    private final ConcurrentMap<String, ImmutableSession> sessionMap;

    HttpSessionManager(boolean central, boolean offlineViewer, ConfigRepository configRepository, Clock clock, LayoutService layoutService, SessionMapFactory sessionMapFactory) {
        this.central = central;
        this.offlineViewer = offlineViewer;
        this.configRepository = configRepository;
        this.clock = clock;
        this.layoutService = layoutService;
        this.sessionMap = sessionMapFactory.create();
    }

    CommonHandler.CommonResponse login(String username, String password) throws Exception {
        if (username.equalsIgnoreCase("anonymous")) {
            HttpSessionManager.auditFailedLogin(username);
            return HttpSessionManager.buildIncorrectLoginResponse();
        }
        UserConfig userConfig = this.getUserConfigCaseInsensitive(username);
        if (userConfig == null || userConfig.ldap()) {
            HashSet roles;
            try {
                roles = this.authenticateAgainstLdapAndGetGlowrootRoles(username, password);
            }
            catch (LdapAuthentication.AuthenticationException e) {
                logger.debug(e.getMessage(), (Throwable)e);
                HttpSessionManager.auditFailedLogin(username);
                return HttpSessionManager.buildIncorrectLoginResponse();
            }
            if (userConfig != null) {
                roles = Sets.newHashSet(roles);
                roles.addAll(userConfig.roles());
            }
            if (!roles.isEmpty()) {
                return this.createSession(username, roles, true);
            }
        } else if (HttpSessionManager.validatePassword(password, userConfig.passwordHash())) {
            return this.createSession(userConfig.username(), (Set<String>)userConfig.roles(), false);
        }
        HttpSessionManager.auditFailedLogin(username);
        return HttpSessionManager.buildIncorrectLoginResponse();
    }

    void signOut(CommonHandler.CommonRequest request) throws Exception {
        Session session;
        String sessionId = this.getSessionId(request);
        if (sessionId != null && (session = (Session)this.sessionMap.remove(sessionId)) != null) {
            HttpSessionManager.auditLogout(session.caseAmbiguousUsername());
        }
    }

    void deleteSessionCookie(CommonHandler.CommonResponse response) throws Exception {
        DefaultCookie cookie = new DefaultCookie(this.configRepository.getWebConfig().toCompletableFuture().join().sessionCookieName(), "");
        cookie.setHttpOnly(true);
        cookie.setMaxAge(0L);
        cookie.setPath("/");
        response.setHeader((CharSequence)HttpHeaderNames.SET_COOKIE, ServerCookieEncoder.STRICT.encode((Cookie)cookie));
    }

    Authentication getAuthentication(CommonHandler.CommonRequest request, boolean touch) throws Exception {
        long timeoutMillis;
        if (this.offlineViewer) {
            return this.getOfflineViewerAuthentication();
        }
        String sessionId = this.getSessionId(request);
        if (sessionId == null) {
            return this.getAnonymousAuthentication();
        }
        Session session = (Session)this.sessionMap.get(sessionId);
        if (session == null) {
            return this.getAnonymousAuthentication();
        }
        long currentTimeMillis = this.clock.currentTimeMillis();
        if (session.isTimedOut(currentTimeMillis, timeoutMillis = TimeUnit.MINUTES.toMillis(this.configRepository.getWebConfig().toCompletableFuture().join().sessionTimeoutMinutes()))) {
            return this.getAnonymousAuthentication();
        }
        if (touch) {
            this.sessionMap.put(sessionId, ImmutableSession.builder().copyFrom(session).lastRequest(currentTimeMillis).build());
        }
        return session.createAuthentication(this.central, this.configRepository);
    }

    @Nullable String getSessionId(CommonHandler.CommonRequest request) throws Exception {
        String cookieHeader = request.getHeader((CharSequence)HttpHeaderNames.COOKIE);
        if (cookieHeader == null) {
            return null;
        }
        Set cookies = ServerCookieDecoder.STRICT.decode(cookieHeader);
        for (Cookie cookie : cookies) {
            if (!cookie.name().equals(this.configRepository.getWebConfig().toCompletableFuture().join().sessionCookieName())) continue;
            return cookie.value();
        }
        return null;
    }

    Authentication getAnonymousAuthentication() throws Exception {
        UserConfig userConfig = this.getUserConfigCaseInsensitive("anonymous");
        return ImmutableAuthentication.builder().central(this.central).offlineViewer(false).anonymous(true).ldap(false).caseAmbiguousUsername("anonymous").roles((Iterable<String>)(userConfig == null ? ImmutableSet.of() : userConfig.roles())).configRepository(this.configRepository).build();
    }

    private CommonHandler.CommonResponse createSession(String username, Set<String> roles, boolean ldap) throws Exception {
        String sessionId = new BigInteger(130, this.secureRandom).toString(32);
        ImmutableSession session = ImmutableSession.builder().caseAmbiguousUsername(username).ldap(ldap).roles(roles).lastRequest(this.clock.currentTimeMillis()).build();
        this.sessionMap.put(sessionId, session);
        String layoutJson = this.layoutService.getLayoutJson(session.createAuthentication(this.central, this.configRepository));
        CommonHandler.CommonResponse response = new CommonHandler.CommonResponse(HttpResponseStatus.OK, MediaType.JSON_UTF_8, layoutJson);
        DefaultCookie cookie = new DefaultCookie(this.configRepository.getWebConfig().toCompletableFuture().join().sessionCookieName(), sessionId);
        cookie.setHttpOnly(true);
        cookie.setPath("/");
        response.setHeader((CharSequence)HttpHeaderNames.SET_COOKIE, ServerCookieEncoder.STRICT.encode((Cookie)cookie));
        this.purgeExpiredSessions();
        HttpSessionManager.auditSuccessfulLogin(username);
        return response;
    }

    private Authentication getOfflineViewerAuthentication() {
        return ImmutableAuthentication.builder().central(false).offlineViewer(true).anonymous(true).ldap(false).caseAmbiguousUsername("anonymous").configRepository(this.configRepository).build();
    }

    private @Nullable UserConfig getUserConfigCaseInsensitive(String username) throws Exception {
        for (UserConfig userConfig : this.configRepository.getUserConfigs().toCompletableFuture().join()) {
            if (!userConfig.username().equalsIgnoreCase(username)) continue;
            return userConfig;
        }
        return null;
    }

    private void purgeExpiredSessions() throws Exception {
        long currentTimeMillis = this.clock.currentTimeMillis();
        long timeoutMillis = TimeUnit.MINUTES.toMillis(this.configRepository.getWebConfig().toCompletableFuture().join().sessionTimeoutMinutes());
        Iterator i = this.sessionMap.entrySet().iterator();
        while (i.hasNext()) {
            Session session = (Session)i.next().getValue();
            if (!session.isTimedOut(currentTimeMillis, timeoutMillis)) continue;
            i.remove();
            HttpSessionManager.auditSessionTimeout(session.caseAmbiguousUsername());
        }
    }

    private Set<String> authenticateAgainstLdapAndGetGlowrootRoles(String username, String password) throws Exception {
        LdapConfig ldapConfig = this.configRepository.getLdapConfig().toCompletableFuture().join();
        String host = ldapConfig.host();
        if (host.isEmpty()) {
            throw new LdapAuthentication.AuthenticationException("LDAP is not configured");
        }
        Set<String> ldapGroupDns = LdapAuthentication.authenticateAndGetLdapGroupDns(username, password, ldapConfig, null, this.configRepository.getLazySecretKey());
        return LdapAuthentication.getGlowrootRoles(ldapGroupDns, ldapConfig);
    }

    private static CommonHandler.CommonResponse buildIncorrectLoginResponse() {
        return new CommonHandler.CommonResponse(HttpResponseStatus.OK, MediaType.JSON_UTF_8, "{\"incorrectLogin\":true}");
    }

    private static void auditFailedLogin(String username) {
        auditLogger.info("{} - failed login", (Object)username);
    }

    private static void auditSuccessfulLogin(String username) {
        auditLogger.info("{} - successful login", (Object)username);
    }

    private static void auditLogout(String username) {
        auditLogger.info("{} - logout", (Object)username);
    }

    private static void auditSessionTimeout(String username) {
        auditLogger.info("{} - session timeout", (Object)username);
    }

    private static boolean validatePassword(String password, String passwordHash) throws GeneralSecurityException {
        if (passwordHash.isEmpty()) {
            return password.isEmpty();
        }
        return PasswordHash.validatePassword(password, passwordHash);
    }

    @Value.Immutable
    static abstract class Authentication {
        Authentication() {
        }

        abstract boolean central();

        abstract boolean offlineViewer();

        abstract boolean anonymous();

        abstract boolean ldap();

        abstract String caseAmbiguousUsername();

        abstract Set<String> roles();

        abstract ConfigRepository configRepository();

        boolean isPermitted(String agentRollupId, String permission) throws Exception {
            if (permission.startsWith("agent:")) {
                return this.isPermittedForAgentRollup(agentRollupId, permission);
            }
            return this.isAdminPermitted(permission);
        }

        boolean isPermittedForAgentRollup(String agentRollupId, String permission) throws Exception {
            Preconditions.checkState((boolean)permission.startsWith("agent:"));
            if (this.offlineViewer()) {
                return !permission.startsWith("agent:config:edit:");
            }
            if (permission.equals("agent:trace")) {
                return this.isPermittedForAgentRollup(agentRollupId, "agent:transaction:traces") || this.isPermittedForAgentRollup(agentRollupId, "agent:error:traces");
            }
            return this.isPermitted(RoleConfig.SimplePermission.create(agentRollupId, permission));
        }

        boolean isAdminPermitted(String permission) throws Exception {
            Preconditions.checkState((boolean)permission.startsWith("admin:"));
            if (this.offlineViewer()) {
                return permission.equals("admin:view") || permission.startsWith("admin:view:");
            }
            return this.isPermitted(RoleConfig.SimplePermission.create(permission));
        }

        boolean hasAnyPermissionImpliedBy(String permission) throws Exception {
            Preconditions.checkState((permission.equals("agent") || permission.startsWith("agent:") ? 1 : 0) != 0);
            if (this.offlineViewer()) {
                return !permission.startsWith("agent:config:edit:");
            }
            List permissionParts = Splitter.on((char)':').splitToList((CharSequence)permission);
            for (RoleConfig roleConfig : this.configRepository().getRoleConfigs().toCompletableFuture().join()) {
                if (!this.roles().contains(roleConfig.name()) || !roleConfig.hasAnyPermissionImpliedBy(permissionParts)) continue;
                return true;
            }
            return false;
        }

        boolean isPermittedForSomeAgentRollup(String permission) throws Exception {
            Preconditions.checkState((boolean)permission.startsWith("agent:"));
            if (this.offlineViewer()) {
                return !permission.startsWith("agent:config:edit:");
            }
            List permissionParts = Splitter.on((char)':').splitToList((CharSequence)permission);
            for (RoleConfig roleConfig : this.configRepository().getRoleConfigs().toCompletableFuture().join()) {
                if (!this.roles().contains(roleConfig.name()) || !roleConfig.isPermittedForSomeAgentRollup(permissionParts)) continue;
                return true;
            }
            return false;
        }

        RoleConfig.HasAnyPermission hasAnyPermissionForAgentRollup(String agentRollupId) throws Exception {
            if (this.offlineViewer()) {
                return RoleConfig.HasAnyPermission.YES;
            }
            boolean onlyInChild = false;
            for (RoleConfig roleConfig : this.configRepository().getRoleConfigs().toCompletableFuture().join()) {
                if (!this.roles().contains(roleConfig.name())) continue;
                RoleConfig.HasAnyPermission hasAnyPermission = roleConfig.hasAnyPermissionForAgentRollup(agentRollupId);
                if (hasAnyPermission == RoleConfig.HasAnyPermission.YES) {
                    return RoleConfig.HasAnyPermission.YES;
                }
                if (hasAnyPermission != RoleConfig.HasAnyPermission.ONLY_IN_CHILD) continue;
                onlyInChild = true;
            }
            return onlyInChild ? RoleConfig.HasAnyPermission.ONLY_IN_CHILD : RoleConfig.HasAnyPermission.NO;
        }

        private boolean isPermitted(RoleConfig.SimplePermission permission) throws Exception {
            for (RoleConfig roleConfig : this.configRepository().getRoleConfigs().toCompletableFuture().join()) {
                if (!this.roles().contains(roleConfig.name()) || !roleConfig.isPermitted(permission)) continue;
                return true;
            }
            return false;
        }
    }

    @Value.Immutable
    @Serial.Structural
    static abstract class Session {
        Session() {
        }

        abstract String caseAmbiguousUsername();

        abstract boolean ldap();

        abstract Set<String> roles();

        abstract long lastRequest();

        Authentication createAuthentication(boolean central, ConfigRepository configRepository) {
            return ImmutableAuthentication.builder().central(central).offlineViewer(false).anonymous(false).ldap(this.ldap()).caseAmbiguousUsername(this.caseAmbiguousUsername()).roles(this.roles()).configRepository(configRepository).build();
        }

        private boolean isTimedOut(long currentTimeMillis, long timeoutMillis) {
            return timeoutMillis != 0L && this.lastRequest() < currentTimeMillis - timeoutMillis;
        }
    }
}

